//------------------------------------------------------------------
//  TestUnitTest.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../util/string";
import { Map } from "../../util/Map";
import { ObjectMap } from "../../util/types";
import { Day } from "../../date/Day";
import { Duration } from "../../date/Duration";
import { UnitTest } from "../UnitTest";
import { UnitTestRunner } from "../UnitTestRunner";
//------------------------------------------------------------------

/** Unit test for the UnitTest class.
 */
export class TestUnitTest extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new UnitTest object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods
  //------------------------------------------------------------------

  /** Checks the error message formats.
   */
  testMessage() : void
  {
    // Wrong types.

    this.check("type", "true", true);
    this.check("type", 3, true);

    // Primitives.

    this.check("number", 3.14, 3.15);
    this.check("boolean", true, false);
    this.check("datetime", new Date(10000), new Date(20000));
    this.check("date", new Day(10000), new Day(20000));
    this.check("duration", new Duration(10000), new Duration(20000));

    // Strings.

    this.check("short string", "a", "b");
    this.check("long string",
               "abcdefghijklmnopqrstuvwxyz1abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz1abcdefghijklmnopqrstuvwxyz",
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz");
    this.check("long strings differ at last character",
               "abcdefghijklmnopqrstuvwxyz1abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz1abcdefghijklmnopqrstuvwxyz1",
               "abcdefghijklmnopqrstuvwxyz1abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz1abcdefghijklmnopqrstuvwxyz2");
    this.check("short and long string",
               "b",
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz");
    this.check("expected string too long",
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz",
               "a");
    this.check("actual string too long",
               "a",
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz" +
               "abcdefghijklmnopqrstuvwxyz2abcdefghijklmnopqrstuvwxyz");

    // Class-less objects.

    let o1 = {"a": 1, "b": 2};
    let o2 = {"a": 1};

    let obj1 = <ObjectMap<any>><any>o1;
    let obj2 = <ObjectMap<any>><any>o2;

    this.check("Object missing key", obj1, obj2);

    obj2['b'] = 2;
    obj2['c'] = 3;
    this.check("Object extra key", obj1, obj2);

    delete obj2['c'];
    obj2['b'] = 2001;

    this.check("Object wrong value", obj1, obj2);

    obj2['b'] = "Orange pips";
    this.check("Object wrong type", obj1, obj2);

    // Objects.

    class Cat
    {
      paws: number = 4;
    }

    class Dog
    {
      teeth: string = "big";
    }

    let cat = new Cat();
    let dog = new Dog();

    this.check("object wrong type", cat, dog);

    let cat2 = new Cat();
    cat2.paws = 5;

    this.check("object wrong property value", cat, cat2);

    // Arrays.

    this.check("array length", [1, 2, 3], [1, 2]);
    this.check("array type", [1, 2, 3], ['1', '2', '3']);

    // Maps.

    let map1: Map<string, number> = Map.createStringMap<number>();
    let map2: Map<string, number> = Map.createStringMap<number>();

    map1.put("a", 1);
    map1.put("b", 2);

    map2.put("a", 1);

    this.check("map missing key", map1, map2);

    map2.put("b", 2);
    map2.put("c", 3);

    this.check("map extra key", map1, map2);
  }

  //------------------------------------------------------------------
  // Private Utility Methods
  //------------------------------------------------------------------

  /** Check that the assert returns an error message.
   */
  private check(msg: string, expected: any, actual: any) : void
  {
    msg += "\n" +  this.checkEqual(expected, actual) + "\n";
    this.write(msg);
  }

  /** Writes to the screen if 'visible' is enabled.
   */
  private write(text: string) : void
  {
    let visible: boolean = false;

    // Don't want to spam the console normally.
    // Un-comment this if you want to see the messages.
    //visible = true;

    if (visible)
    {
      console.log(text);
    }
  }

} // END class TestUnitTest

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestUnitTest());
