//------------------------------------------------------------------
//  TestJsonPath.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../../../util/string';
import { UnitTest } from '../../../unittest/UnitTest';
import { UnitTestRunner } from '../../../unittest/UnitTestRunner';
import { JsonPath } from '../JsonPath';
//------------------------------------------------------------------

/** Unit test for the JsonPath class.
 */
export class TestJsonPath extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonPath object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** Test the parse() static constructor.
   */
  testParsing() : void
  {
    let path = JsonPath.parse('../a/b[42]/c');

    this.assertEqual('elements 0', '..', path.elements[0].name);
    this.assertEqual('elements 1', 'a', path.elements[1].name);
    this.assertEqual('elements 2', 'b', path.elements[2].name);
    this.assertEqual('elements 3', 'c', path.elements[3].name);

    this.assertEqual('elements 0', null, path.elements[0].index);
    this.assertEqual('elements 1', null, path.elements[1].index);
    this.assertEqual('elements 2', 42, path.elements[2].index);
    this.assertEqual('elements 3', null, path.elements[3].index);

    // Illegal input is rejected.

    try
    {
      JsonPath.parse('/../a/b/c');
      this.fail("No exception for a leading slash.");
    }
    catch
    {
    }

    // String conversion.

    // Don't check format, just make sure no exception is thrown.
    path.toString();

    this.assertEqual('pathString', '../a/b[42]/c', path.pathString());
  }

  /** Test resolving a path.
   */
  testResolving() : void
  {
    // A tree of objects for resolving.

    class Root
    {
      a: A = new A();
    }

    class A
    {
      b1: B1 = new B1();
      b2: B2 = new B2();
    }

    class B1
    {
      c1a: C1a = new C1a();
      c1b: C1b = new C1b();
    }

    class B2
    {
      c2a: C2a = new C2a();
      c2b: C2b = new C2b();
    }

    class C1a
    {
    }

    class C1b
    {
    }

    class C2a
    {
    }

    class C2b
    {
    }

    let root = new Root();

    // Resolve from the top down.

    let expected: object = root.a.b2.c2a;
    let actual: object = JsonPath.parse('a/b2/c2a').resolve([], root);

    this.assertEqual('a/b2/c2a', expected, actual);

    // Resolve from the bottom up.

    let parents: Array<object> = [root,
                                  root.a,
                                  root.a.b2];

    expected = root;
    actual = JsonPath.parse('../../..').resolve(parents, root.a.b2.c2a);

    this.assertEqual('../../..', expected, actual);

    // Go up and down to find a sibling.

    parents = [root,
               root.a,
               root.a.b2];

    expected = root.a.b1.c1b;
    actual = JsonPath.parse('../../../a/b1/c1b').resolve(parents,
                                                         root.a.b2.c2a);

    this.assertEqual('../../../a/b1/c1b', expected, actual);

    // An empty path resolves to the start object.

    expected = root;
    actual = (new JsonPath([])).resolve([], root);

    this.assertEqual('empty path', expected, actual);

    // Try to go up past the root.

    parents = [root,
               root.a,
               root.a.b2];

    try
    {
      JsonPath.parse('../../../..').resolve(parents, root.a.b2.c2a);
      this.fail("No exception going up past the root.");
    }
    catch (ex)
    {
      this.assertEqual('going up past the root',
                       true,
                       ex.toString().includes('no parents'));
    }
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestJsonPath

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonPath());
