//------------------------------------------------------------------
//  TestJsonDateTime.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../../util/string";
import { UnitTest } from "../../../unittest/UnitTest";
import { UnitTestRunner } from "../../../unittest/UnitTestRunner";
import { JsonDateTime } from "../JsonDateTime";
//------------------------------------------------------------------

/** Unit test for the JsonDateTime class.
 */
export class TestJsonDateTime extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonDateTime object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** Test the encode/decode() method.
   */
  testEncodeDecode() : void
  {
    let jsonType = new JsonDateTime();

    // Validate typescript.

    this.assertEqual('date valid',
                     null, jsonType.validate(new Date(2016, 0, 2)));
    this.assertEqual('null valid', null, jsonType.validate(null));

    this.assertNotNull('string invalid', jsonType.validate('invalid'));
    this.assertNotNull('float invalid', jsonType.validate(3.14));

    // Validate JSON.

    this.assertEqual('date valid JSON',
                     null,
                     jsonType.validateJson("2016-01-01T01:01:34.123Z"));
    this.assertEqual('null valid JSON', null, jsonType.validateJson(null));

    this.assertNotNull('string invalid', jsonType.validateJson('invalid'));
    this.assertNotNull('float invalid', jsonType.validateJson(3.14));

    // Encode.

    this.assertEqual('encode',
                     "2016-01-02T03:04:05.678Z",
                     jsonType.encode(this.makeDate(2016, 0, 2, 3, 4, 5, 678)));
    this.assertEqual('encode no millis',
                     "2016-01-02T03:04:05.000Z",
                     jsonType.encode(new Date(Date.UTC(2016, 0, 2, 3, 4, 5))));
    this.assertEqual('null', null, jsonType.encode(null));

    // Decode.

    this.assertEqual('decode',
                     this.makeDate(2016, 0, 2, 3, 4, 5, 678),
                     jsonType.decode("2016-01-02T03:04:05.678Z"));
    this.assertEqual('decode no millis',
                     new Date(Date.UTC(2016, 0, 2, 3, 4, 5)),
                     jsonType.decode("2016-01-02T03:04:05.000Z"));
    this.assertEqual('null', null, jsonType.decode(null));

    // Decode links.

    this.assertEqual('decode links',
                     new Date(2016, 1, 2, 3, 4, 5),
                     jsonType.decodeLinks([], new Date(2016, 1, 2, 3, 4, 5)));

    this.assertEqual('null', null, jsonType.decodeLinks([], null));
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

  private makeDate(year: number,
                   month: number,
                   day: number,
                   hours: number,
                   minutes: number,
                   seconds: number,
                   ms: number) : Date
  {
    return new Date(Date.UTC(year, month, day, hours, minutes, seconds, ms));
  }

} // END class TestJsonDateTime

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonDateTime());
