//------------------------------------------------------------------
//  JsonLink.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeArray from '../../util/array';
import * as axeClasses from '../../util/classes';
import * as axeString from '../../util/string';
import { ClassJsonDesc } from '../ClassJsonDesc';
import { ClassJsonRegistry } from '../ClassJsonRegistry';
import { JsonObj }  from './JsonObj';
import { JsonPath }  from '../link/JsonPath';
import { JsonType }  from './JsonType';
import { JsonTypeError } from './JsonTypeError';
import { Jsonable } from '../classJson';
import { ObjectMap } from '../../util/types';
import { assert }  from '../classJson';
//------------------------------------------------------------------

/** Attribute that is a pointer to an object somewhere else in the JSON tree.
 */
export class JsonLink extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The class of the object this is a link to.  Must be a ClassJsonClass.
   */
  objClassName: string;

  /** The JsonObj for the object that this is a link to.
   */
  linkedJsonObj: JsonObj;

  /** The path to find the object this is a link to.
   */
  objPath: JsonPath;

  /** The member variable to use to find the correct object in the collection.
   */
  idFields: Array<string>;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonLink object.
   */
  constructor(objClassName: string,
              objPath: string,
              idFields: Array<string>)
  {
    super();

    this.objClassName = objClassName;
    this.objPath = JsonPath.parse(objPath);
    this.idFields = idFields;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Returns the class of the object that this encodes/decodes.
   */
  objClass() : Jsonable<any, any>
  {
    return ClassJsonRegistry.registry.registeredClass(this.objClassName);
  }

  /** Returns the description of the object that this encodes/decodes.
   */
  objDesc() : ClassJsonDesc
  {
    return ClassJsonRegistry.registry.getDesc(this.objClassName);
  }

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    if (!(value instanceof <any>(this.objClass())))
    {
      let msg = ('is of type ' + this.objClassName  +
                 ' but the value is of type ' +
                 axeClasses.className(value) + '.  Value: ' + value);
      return new JsonTypeError(msg);
    }

    for (let idField of this.idFields)
    {
      if (typeof (<ObjectMap<any>>value)[idField] == 'undefined')
      {
        let msg = ('uses link fields ' + axeString.format(this.idFields) +
                   ' but the value is missing field "' + idField + '".  ' +
                   'Value: ' + value);
        return new JsonTypeError(msg);
      }
    }

    return null;
  }


  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }
    else if (!(value instanceof Array))
    {
      let msg = ('is link ID field ' + axeString.format(this.idFields) +
                 ', but has invalid type "' + axeClasses.className(value) +
                 '". ' + 'Value: ' + value);
      return new JsonTypeError(msg);
    }

    for (let item of value)
    {
      if (!(typeof item == 'number' || typeof item == 'string'))
      {
        let msg = ('is link ID field ' + axeString.format(this.idFields) +
                   ', but has invalid type "' + axeClasses.className(item) +
                   '". ' + 'Value: ' + value);
        return new JsonTypeError(msg);
      }
    }

    return null;
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    // Return the ID value that can be used to find the object at decode time.
    let idFieldsValue: Array<object> = [];

    for (let idField of this.idFields)
    {
      assert(typeof value[idField] != 'undefined');
      idFieldsValue.push(value[idField]);
    }

    return idFieldsValue;
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    // Return the ID value unchanged.  We will go find the actual object later.
    return value;
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    // The 'value' is an attribute of an object, so it should have
    // at least one parent, the object it belongs to.
    assert(parents.length > 0);

    if (value === null)
    {
      return value;
    }

    // The value is an array of idField values.
    assert(value instanceof Array);
    let valueArray = <Array<any>>value;

    let linkedObj = this.objPath.resolve(parents.slice(0, -1),
                                         axeArray.last(parents));

    if (!axeArray.isArray(linkedObj))
    {
      let msg = ('Link ' + this.objPath.pathString() + ' expects a list of ' +
                 this.objClassName + ' objects, but instead ' +
                 'found ' + axeClasses.className(linkedObj)  + ' object: ' +
                 linkedObj);
      throw new Error(msg);
    }

    let linkedArray = <Array<any>>linkedObj;

    for (let i = 0; i < linkedArray.length; ++i)
    {
      let item = linkedArray[i];

      if (!(item instanceof <any>(this.objClass())))
      {
        let msg = ('Link ' + this.objPath.pathString() +
                   ' expects a list of type ' + this.objClassName +
                   ', but item ' + i + ' has type ' +
                   axeClasses.className(item) +
                   '.  Value: ' + item);
        throw new Error(msg);
      }

      // Initialize to false if the array is empty or missing values,
      // as we won't check the change to check anything in the loop below.
      let isIdMatch = (this.idFields.length > 0 &&
                       this.idFields.length == valueArray.length);

      for (let j = 0; j < this.idFields.length; ++j)
      {
        let idField = this.idFields[j];
        let valueItem = valueArray[j];

        if (item[idField] != valueItem)
        {
          isIdMatch = false;
        }
      }

      if (isIdMatch)
      {
        return item;
      }
    }

    let errMsg = ('Link ' + this.objPath.pathString() + ' failed to find ' +
                  'an object with ' + axeString.format(this.idFields) + ' = ' +
                  value + '.');
    throw new Error(errMsg);
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      "objClassName",
      "objPath",
      "idFields",
    ];
    return axeString.formatObject("JsonLink", this, propertyNames);
  }

} // END class JsonLink
