//------------------------------------------------------------------
//  ClassJsonInterceptor.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../util/string';
import { ClassJsonDecoder } from './ClassJsonDecoder';
import { HTTP_INTERCEPTORS } from '@angular/common/http';
import { HttpEvent } from '@angular/common/http';
import { HttpHandler } from '@angular/common/http';
import { HttpInterceptor } from '@angular/common/http';
import { HttpRequest } from '@angular/common/http';
import { HttpResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
//------------------------------------------------------------------

/** Intercepts HTTP responses to convert JSON to classes.
 */
@Injectable()
export class ClassJsonInterceptor implements HttpInterceptor
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  decoder: ClassJsonDecoder = new ClassJsonDecoder();

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonInterceptor object.
   */
  constructor()
  {
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  intercept(request: HttpRequest<any>,
            next: HttpHandler) : Observable<HttpEvent<any>>
  {
    let observable: Observable<HttpEvent<any>> = next.handle(request);

    observable = observable.pipe(map(this.modifyResponse.bind(this)));

    return observable;
  }

  /** Modifies the response body from JSON to an object.
   */
  modifyResponse(event: HttpEvent<any>) : HttpEvent<any>
  {
    if (event instanceof HttpResponse)
    {
      let response: HttpResponse<any> = event as HttpResponse<any>;

      // If the response is JSON, decode it.

      // The type will be 'object' because Angular has already called
      // JSON.parse() on the response body by the time we get it.

      let contentType = response.headers.get('Content-Type');
      if (contentType && (contentType.indexOf('application/json') === 0))
      {
        if (typeof response.body != 'object')
        {
          console.error("Expected object as HTTP response, but received ",
                        "non-string object: ", response.body);
          throw new Error("Expected object as HTTP response, but received " +
                          "non-object data: " + response.body);
        }

        let attrs = {"body": this.decoder.decodeObject(response.body)};
        response = response.clone(<any>attrs);

        return response;
      }
    }
    return event;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("ClassJsonInterceptor", this, propertyNames);
  }

} // END class ClassJsonInterceptor


/** To enable this class, add this to 'providers' in the app module set up.
 */
export const classJsonProvider = {
  "provide": HTTP_INTERCEPTORS,
  "useClass": ClassJsonInterceptor,
  "multi": true,
};
