//------------------------------------------------------------------
//  ClassJsonDesc.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
/// <reference path="../util/types" />
import { ClassJsonRegistry } from './ClassJsonRegistry';
import { JsonType } from "./jsonTypes/JsonType";
import { Map } from '../util/Map';
import { ObjectMap } from '../util/types';
//------------------------------------------------------------------

/** A description of a single ClassJson class schema.
 */
export class ClassJsonDesc
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The name of the class.
   */
  className: string;

  /** The fields and their types.
   */
  private fields: Map<string, JsonType> = Map.createStringMap<JsonType>();

  /** True if this is an abstract class.
   */
  isAbstract: boolean = false;

  /** The base class of this.
   */
  baseClass: string = null;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonDesc object.
   */
  constructor(className: string)
  {
    this.className = className;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Sets whether this is an abstract class.
   */
  setAbstract(isAbstract: boolean) : void
  {
    this.isAbstract = isAbstract;
  }

  /** Sets the base class of this class.  This is a JSON _class name.
   */
  setBaseClass(className: string) : void
  {
    this.baseClass = className;
  }

  /** Gets a list of all bases of this class.
   */
  baseClassNames() : Array<string>
  {
    let names: Array<string> = [];

    let name = this.baseClass;

    while (name)
    {
      names.push(name);
      name = ClassJsonRegistry.registry.getDesc(name).baseClass;
    }

    return names;
  }

  /** Gets the names of all the fields.
   */
  getFieldNames() : Array<string>
  {
    return this.fields.keys();
  }

  /** Returns the type of the specified field.
   */
  getFieldType(field: string) : JsonType
  {
    if (!this.fields.hasKey(field))
    {
      let msg: string = "Error: class '" + this.className + "' has no field: ";
      msg += field;
      throw new Error(msg);
    }
    return this.fields.get(field);
  }

  /** Adds a field.
   */
  addField(field: string, fieldType: JsonType) : void
  {
    this.fields.put(field, fieldType);
  }

  /** Adds all the fields in an object.
   */
  addFieldsObject(fields: ObjectMap<JsonType>) : void
  {
    for (let key in fields)
    {
      this.fields.put(key, fields[key]);
    }
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      "className",
      "fields",
    ];
    return axeString.formatObject("ClassJsonDesc", this, propertyNames);
  }

} // END class ClassJsonDesc
