#-------------------------------------------------------------------
#  WktTestRaster.py
#
#  The WktTestRaster class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WktTestRaster class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.wkt import Raster
from ai.axe.wkt import WkbRasterReader
from ai.axe.wkt import WkbRasterWriter
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WktTestRaster:
  '''An object that holds information about a single test raster object.
  '''

  #-----------------------------------------------------------------
  def __init__(self, name: str,
               obj: Raster,
               wkbBig: str,
               wkbLittle: str) -> None:
    '''Creates a new WktTestRaster.
    '''

    # The name of this test to be used in output to the user.
    self.name = name

    # The raster object being converted from/to wkb.
    self.obj = obj

    # The big-endian WKB form of the object.
    self.wkbBig = wkbBig

    # The little-endian WKB form of the object.
    self.wkbLittle = wkbLittle

  #-----------------------------------------------------------------
  def runReadTest(self, testObj: AxeSimpleTestCase) -> None:
    '''Runs a test of reading this test raster.
    '''

    reader = WkbRasterReader()
    obj = reader.read(self.wkbLittle)

    self.assertObj(testObj, obj, 'Reading ' + self.name + ' little-endian')

    reader = WkbRasterReader()
    obj = reader.read(self.wkbBig)

    self.assertObj(testObj, obj, 'Reading ' + self.name + ' big-endian')

  #-----------------------------------------------------------------
  def runWriteTest(self, testObj: AxeSimpleTestCase) -> None:
    '''Runs a test of writing this test raster.
    '''

    writer = WkbRasterWriter('little')
    data = writer.write(self.obj)

    testObj.assertEqual(self.wkbLittle.lower(), data,
                        'Writing ' + self.name + ' little-endian')

    writer = WkbRasterWriter('big')
    data = writer.write(self.obj)

    testObj.assertEqual(self.wkbBig.lower(), data,
                        'Writing ' + self.name + ' big-endian')

  #-----------------------------------------------------------------
  def assertObj(self,
                testObj: AxeSimpleTestCase,
                obj: Raster,
                msg: str) -> None:
    '''Asserts that the 'obj' is the same as self.obj.

    @param testObj A Unittest object to raise test exceptions.
    @param obj The object to be checked.
    '''

    testObj.assertEqual(self.obj.width, obj.width,
                        msg + ' width')
    testObj.assertEqual(self.obj.height, obj.height,
                        msg + ' height')
    testObj.assertEqual(self.obj.upperLeftX, obj.upperLeftX,
                        msg + ' upperLeftX')
    testObj.assertEqual(self.obj.upperLeftY, obj.upperLeftY,
                        msg + ' upperLeftY')
    testObj.assertEqual(self.obj.scaleX, obj.scaleX,
                        msg + ' scaleX')
    testObj.assertEqual(self.obj.scaleY, obj.scaleY,
                        msg + ' scaleY')
    testObj.assertEqual(self.obj.skewX, obj.skewX,
                        msg + ' skewX')
    testObj.assertEqual(self.obj.skewY, obj.skewY,
                        msg + ' skewY')
    testObj.assertEqual(self.obj.srid, obj.srid,
                        msg + ' srid')

    testObj.assertEqual(len(self.obj.bands), len(obj.bands), msg + ' # bands')

    for i in range(len(self.obj.bands)):

      selfBand = self.obj.bands[i]
      objBand = obj.bands[i]

      testObj.assertEqual(selfBand.pixelType, objBand.pixelType,
                          msg + ' pixelType')
      testObj.assertEqual(selfBand.noData, objBand.noData,
                          msg + ' noData')
      testObj.assertEqual(selfBand.isAllNoData, objBand.isAllNoData,
                          msg + ' isAllNoData')
      testObj.assertEqual(selfBand.values, objBand.values,
                          msg + ' values')
