#-------------------------------------------------------------------
#  TestWktReader.py
#
#  The TestWktReader module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the WktReader class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.wkt import WktReader
from ai.axe.build.unittest import AxeSimpleTestCase
from . import WktTestData
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestWktReader(AxeSimpleTestCase):
  '''Unit test for the WktReader class.
'''

  #-----------------------------------------------------------------
  def setUp(self):

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self):

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testReading(self):
    '''Test reading a geometry.
    '''
    names = ['point',
             'linestring',
             'polygon_1ring',
             'polygon_2ring',
             'polygon_3ring',
             'multipolygon_1',
             'multipolygon_2']

    for name in names:
      testObj = WktTestData.getTestObject(name)
      testObj.runReadTextTest(self)

  #-----------------------------------------------------------------
  def testBadInput(self):
    '''Test errors for malformed input.
    '''

    reader = WktReader()

    missingOpenParen = "POINT"

    try:
      reader.read(missingOpenParen)
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass

    missingCloseParen = "POINT(1 2"

    try:
      reader.read(missingCloseParen)
      self.fail("No exception for missing close paren.")
    except ValueError:
      pass

    invalidType = "BADTYPE(1 2)"

    try:
      reader.read(invalidType)
      self.fail("No exception for invalid type.")
    except ValueError:
      pass

  #-----------------------------------------------------------------
  def testParsePoint(self) -> None:
    '''Test converting a POINT string to a python (x, y) tuple.
    '''

    text = "POINT(0 1)"
    expected: Tuple[float, float] = (0, 1)
    actual = WktReader().parsePoint(text)

    self.assertEqual(expected, actual, "point")

    text = "POINT (2.1 3.2)"
    expected = (2.1, 3.2)
    actual = WktReader().parsePoint(text)

    self.assertEqual(expected, actual, "float point")

    try:
      WktReader().parsePoint("POINT(0 1")
      self.fail("No exception for bad ending.")
    except ValueError:
      pass

    try:
      WktReader().parsePoint("BAD(0 1)")
      self.fail("No exception for bad start.")
    except ValueError:
      pass

    try:
      WktReader().parsePoint("POINT 0 1)")
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass

  #-----------------------------------------------------------------
  def testParsePoint3d(self) -> None:
    '''Test converting a POINT Z string to a python (x, y, z) tuple.
    '''

    text = "POINT Z(0 1 2)"
    expected: Tuple[float, float, float] = (0, 1, 2)
    actual = WktReader().parsePoint3d(text)

    self.assertEqual(expected, actual, "point")

    text = "POINT Z (2.1 3.2 4.3)"
    expected = (2.1, 3.2, 4.3)
    actual = WktReader().parsePoint3d(text)

    self.assertEqual(expected, actual, "float point")

    try:
      WktReader().parsePoint3d("POINT Z(0 1 2")
      self.fail("No exception for bad ending.")
    except ValueError:
      pass

    try:
      WktReader().parsePoint3d("BAD(0 1 2)")
      self.fail("No exception for bad start.")
    except ValueError:
      pass

    try:
      WktReader().parsePoint3d("POINT Z 0 1 2)")
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass

  #-----------------------------------------------------------------
  def testParsePoloygon(self) -> None:
    '''Test converting a POLYGON string to a python array.
    '''

    # Simple poloygon.

    raw = '''POLYGON ((-90.07 39.91, -90.08 39.92,-90.09 39.93))'''

    expected1 = [[(-90.07, 39.91), (-90.08, 39.92), (-90.09, 39.93)]]

    actual1 = WktReader().parsePolygon(raw)

    self.assertEqual(expected1, actual1, 'POLYGON')

    # Nested poloygons.

    raw = ('POLYGON ((-90.07 39.91, -90.08 39.92,-90.09 39.93),' +
           '(-90.07 39.91, -90.08 39.92,-90.09 39.93))')

    expected1 = [[(-90.07, 39.91), (-90.08, 39.92), (-90.09, 39.93)],
                [(-90.07, 39.91), (-90.08, 39.92), (-90.09, 39.93)]]

    actual1 = WktReader().parsePolygon(raw)

    self.assertEqual(expected1, actual1, 'POLYGON')

    # Simple multi-poloygon.

    raw = ('MULTIPOLYGON ' +
           '(((-90.07 39.91, -90.08 39.92,-90.09 39.93), ' +
           '(1 1, 1 2,2 2)))')

    expected = [[[(-90.07, 39.91), (-90.08, 39.92), (-90.09, 39.93)],
                 [(1, 1), (1, 2), (2, 2)]]]

    actual = WktReader().parseMultiPolygon(raw)

    self.assertEqual(expected, actual, 'MULTIPOLYGON')

    # Nested multi-poloygon.

    raw = ('MULTIPOLYGON ' +
           '(((-90.07 39.91, -90.08 39.92,-90.09 39.93)), ' +
           '((-90.07 39.91, -90.08 39.92,-90.09 39.93), ' +
           '(1 1, 1 2,2 2)))')

    expected = [[[(-90.07, 39.91), (-90.08, 39.92), (-90.09, 39.93)]],
                [[(-90.07, 39.91), (-90.08, 39.92), (-90.09, 39.93)],
                 [(1, 1), (1, 2), (2, 2)]]]

    actual = WktReader().parseMultiPolygon(raw)

    self.assertEqual(expected, actual, 'MULTIPOLYGON')

  #-----------------------------------------------------------------
  def testParsePoloygonErrors(self) -> None:
    '''Test exceptions when converting a POLYGON string to a python array.
    '''

    # Missing open paren.

    raw = '''POLYGON (-90.07 39.91, -90.08 39.92,-90.09 39.93))'''

    try:
      WktReader().parsePolygon(raw)
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass

    # Missing close paren.

    raw = '''POLYGON ((-90.07 39.91, -90.08 39.92,-90.09 39.93)'''

    try:
      WktReader().parsePolygon(raw)
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass

    # Missing open paren.

    raw = '''MULTIPOLYGON ((-90.07 39.91, -90.08 39.92,-90.09 39.93)))'''

    try:
      WktReader().parseMultiPolygon(raw)
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass

    # Missing close paren.

    raw = '''MULTIPOLYGON (((-90.07 39.91, -90.08 39.92,-90.09 39.93))'''

    try:
      WktReader().parseMultiPolygon(raw)
      self.fail("No exception for missing open paren.")
    except ValueError:
      pass
