#-------------------------------------------------------------------
#  WktWriter.py
#
#  The WktWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WktWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .GeoShape import GeoShape
from .LineString import LineString
from .Point import Point
from .Point3d import Point3d
from .Polygon import Polygon
from .PolygonRing import PolygonRing
from .MultiPolygon import MultiPolygon
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WktWriter:
  '''Writes a python object into Well-Known Text data.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WktWriter.
    '''

    pass

  #-----------------------------------------------------------------
  def write(self, obj: GeoShape) -> str:
    '''Writes the object and returns the encoded data.

    @param obj a python object.

    @return A string of WKT-encoded geometry data.
    '''

    data = ''

    if isinstance(obj, Point):
      data += "POINT("
      data += self.writePoint(obj)
      data += ")"
    elif isinstance(obj, Point3d):
      data += "POINT Z("
      data += self.writePoint3(obj)
      data += ")"
    elif isinstance(obj, LineString):
      data += "LINESTRING"
      data += self.writePolygonRing(obj)
    elif isinstance(obj, Polygon):
      data += "POLYGON"
      data += self.writePolygon(obj)
    elif isinstance(obj, MultiPolygon):
      data += "MULTIPOLYGON"
      data += self.writeMultiPolygon(obj)
    else:
      msg = "Unsupported geometry type: " + str(obj.__class__)
      raise TypeError(msg)

    return data

  #-----------------------------------------------------------------
  def writePoint(self, point: Point) -> str:
    '''Returns the binary data encoding for an object.

    @param point The Point object to encode.

    @return The binary data.
    '''

    return "%s %s" % (point.x, point.y)

  #-----------------------------------------------------------------
  def writePoint3(self, point: Point3d) -> str:
    '''Returns the binary data encoding for an object.

    @param point The Point object to encode.

    @return The binary data.
    '''

    return "%s %s %s" % (point.x, point.y, point.z)

  #-----------------------------------------------------------------
  def writePolygonRing(self, polygonRing: PolygonRing) -> str:
    '''Returns the binary data encoding for an object.

    @param polygonRing The PolygonRing object to encode.

    @return The binary data.
    '''

    pointStrs = []

    for point in polygonRing.points:
      pointStrs.append(self.writePoint(point))

    ret = "(" + ",".join(pointStrs) + ")"

    return ret

  #-----------------------------------------------------------------
  def writePolygon(self, polygon: Polygon) -> str:
    '''Returns the binary data encoding for an object.

    @param polygon The Polygon object to encode.

    @return The binary data.
    '''

    ringStrs = []

    for ring in polygon.rings:
      ringStrs.append(self.writePolygonRing(ring))

    ret = "(" + ",".join(ringStrs) + ")"

    return ret

  #-----------------------------------------------------------------
  def writeMultiPolygon(self, multiPolygon: MultiPolygon) -> str:
    '''Returns the binary data encoding for an object.

    @param multiPolygon The MultiPolygon object to encode.

    @return The binary data.
    '''

    polygonStrs = []

    for polygon in multiPolygon.polygons:
      polygonStrs.append(self.writePolygon(polygon))

    ret = "(" + ",".join(polygonStrs) + ")"

    return ret
