#-------------------------------------------------------------------
#  WktReader.py
#
#  The WktReader class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WktReader class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .GeoShape import GeoShape
from .LineString import LineString
from .Point import Point
from .Point3d import Point3d
from .PolygonRing import PolygonRing
from .Polygon import Polygon
from .MultiPolygon import MultiPolygon
from typing import List
from typing import Tuple
import re
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WktReader:
  '''Reads Well-Known Binary data into a python object.
  '''

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new WktReader.
    '''

    pass

  #-----------------------------------------------------------------
  def read(self, dataStr: str) -> GeoShape:
    '''Reads the binary data and returns an object.

    @param dataStr A WKT string.

    @return a python object.
    '''

    data = dataStr.strip()

    startIndex = data.find('(')
    endIndex = len(data)

    # Read the type.

    if startIndex < 0:
      msg = "WTK string has no opening paren.  Data: " + str(dataStr)
      raise ValueError(msg)

    if not data.endswith(')'):
      msg = "WTK string has no close paren.  Data: " + str(dataStr)
      raise ValueError(msg)

    geoType = data[:startIndex]
    geoType = geoType.strip()

    startIndex += 1
    endIndex -= 1

    if geoType == 'POINT Z':
      point3Datas = self.parsePoint3List(data[startIndex:endIndex])
      x, y, z = point3Datas[0]
      return Point3d(x, y, z)

    if geoType == 'POINT':
      pointDatas = self.parsePointList(data[startIndex:endIndex])
      x, y = pointDatas[0]
      return Point(x, y)

    elif geoType == 'LINESTRING':
      pointDatas = self.parsePointList(data[startIndex:endIndex])
      points = [Point(x, y) for x, y in pointDatas]
      return LineString(points)

    elif geoType == 'POLYGON':
      ringDatas = self.parsePolygon(data)
      rings = []
      for pointDatas in ringDatas:
        points = []
        for pointData in pointDatas:
          x, y = pointData
          points.append(Point(x, y))
        rings.append(PolygonRing(points))

      return Polygon(rings)

    elif geoType == 'MULTIPOLYGON':
      polygons = []
      polygonDatas = self.parseMultiPolygon(data)
      for ringDatas in polygonDatas:
        rings = []
        for pointDatas in ringDatas:
          points = []
          for pointData in pointDatas:
            x, y = pointData
            points.append(Point(x, y))
          rings.append(PolygonRing(points))
        polygons.append(Polygon(rings))

      return MultiPolygon(polygons)

    else:
      msg = ("Unsupported geometry type: " + str(geoType) +
             " Data: " + str(dataStr))
      raise ValueError(msg)

  #-------------------------------------------------------------------
  def parsePoint(self, text: str) -> Tuple[float, float]:
    '''Parses a geospatial WKR POINT.

    @param text The text to parse.

    @return A tuple of (x, y) values.
    '''

    if not text.startswith("POINT"):
      raise ValueError("String doesn't start with POINT. Value: " + str(text))

    if not text.endswith(")"):
      raise ValueError("String doesn't end with paren. Value: " + str(text))

    valueText = text[len("POINT"):]
    valueText = valueText.strip()

    if not valueText.startswith("("):
      raise ValueError("String doesn't start with (. Value: " + str(text))

    pointDatas = self.parsePointList(valueText[len("("):-len(")")])
    x, y = pointDatas[0]

    return x, y

  #-------------------------------------------------------------------
  def parsePoint3d(self, text: str) -> Tuple[float, float, float]:
    '''Parses a geospatial WKR POINT Z.

    @param text The text to parse.

    @return A tuple of (x, y, z) values.
    '''

    if not text.startswith("POINT Z"):
      raise ValueError("String doesn't start with POINT Z. Value: " + str(text))

    if not text.endswith(")"):
      raise ValueError("String doesn't end with paren. Value: " + str(text))

    valueText = text[len("POINT Z"):]
    valueText = valueText.strip()

    if not valueText.startswith("("):
      raise ValueError("String doesn't start with (. Value: " + str(text))

    pointDatas = self.parsePoint3List(valueText[len("("):-len(")")])
    x, y, z = pointDatas[0]

    return x, y, z

  #-------------------------------------------------------------------
  def parsePolygon(self, text: str) -> List[List[Tuple[float, float]]]:
    '''Parses a geospatial WKT POLOYGON.

    @param text The text to parse.

    @return A list of polygons, where a ploygon is a list of points, and
            a point is a list of floats.
    '''

    polygonText = text

    text = text.strip()

    try:

      assert text.upper().startswith('POLYGON')

      text = text[len('POLYGON'):]
      text = text.strip()

      if not text.startswith('(('):
        msg = 'Error parsing POLYGON: does not begin with ((.'
        raise ValueError(msg)
      if not text.endswith('))'):
        msg = 'Error parsing POLYGON: does not end with )).'
        raise ValueError(msg)

      # Remove the leading and trailing parens.
      text = text[2:-2]

      polygons = []

      polygonStrs = re.split(r'\),\s*\(', text)
      for polygonStr in polygonStrs:

        polygons.append(self.parsePointList(polygonStr))

      return polygons

    except ValueError as ex:

      msg = (str(ex) +
             "  Error while processing polygon string: " + polygonText)
      raise ValueError(msg)

  #-------------------------------------------------------------------
  def parseMultiPolygon(self,
                        text: str) -> List[List[List[Tuple[float, float]]]]:
    '''Parses a geospatial WKT MULTIPOLYGON.

    @param text The text to parse.

    @return A list of polygons, where a ploygon is a list of points, and
            a point is a list of floats.
    '''

    polygonText = text

    text = text.strip()

    try:

      assert text.upper().startswith('MULTIPOLYGON')

      text = text[len('MULTIPOLYGON'):]
      text = text.strip()

      if not text.startswith('((('):
        msg = 'Error parsing MULTIPOLYGON: does not begin with (((.'
        raise ValueError(msg)
      if not text.endswith(')))'):
        msg = 'Error parsing MULTIPOLYGON: does not end with ))).'
        raise ValueError(msg)

      # Remove the leading and trailing parens.
      text = text[3:-3]

      multiPolygons = []

      # Split the text into multi-poloygons.

      multiPolygonStrs = re.split(r'\)\),\s*\(\(', text)
      for multiPolygonStr in multiPolygonStrs:

        polygons = []

        # Split the text into poloygons.

        polygonStrs = re.split(r'\),\s*\(', multiPolygonStr)
        for polygonStr in polygonStrs:

          polygons.append(self.parsePointList(polygonStr))

        multiPolygons.append(polygons)

      return multiPolygons

    except ValueError as ex:

      msg = (str(ex) +
             "  Error while processing polygon string: " + polygonText)
      raise ValueError(msg)

  #-------------------------------------------------------------------
  def parsePointList(self, text: str) -> List[Tuple[float, float]]:
    '''Parses a sting of the format: N N, N N, N N

    @param text The text to parse.
    @param sep The seperator to use to split the points.

    @return A list of (x, y) tuples of floats.
    '''

    text = text.strip()

    points = []

    # Split the text into points.

    pointStrs = text.split(',')
    for pointStr in pointStrs:

      # Split the points into 2 floats.

      pointStr = pointStr.strip()
      xStr, yStr = pointStr.split(' ')
      x = float(xStr)
      y = float(yStr)

      points.append((x, y))

    return points

  #-------------------------------------------------------------------
  def parsePoint3List(self, text: str) -> List[Tuple[float, float, float]]:
    '''Parses a sting of the format: N N N, N N N, N N N

    @param text The text to parse.
    @param sep The seperator to use to split the points.

    @return A list of (x, y, z) tuples of floats.
    '''

    text = text.strip()

    points = []

    # Split the text into points.

    pointStrs = text.split(',')
    for pointStr in pointStrs:

      # Split the points into 3 floats.

      pointStr = pointStr.strip()
      xStr, yStr, zStr = pointStr.split(' ')
      x = float(xStr)
      y = float(yStr)
      z = float(zStr)

      points.append((x, y, z))

    return points
