#-------------------------------------------------------------------
#  WkbWriter.py
#
#  The WkbWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WkbWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .LineString import LineString
from .Point import Point
from .Point3d import Point3d
from .Polygon import Polygon
from .MultiPolygon import MultiPolygon
import binascii
import struct
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WkbWriter:
  '''Writes a python object into Well-Known Binary data.
  '''

  #-----------------------------------------------------------------
  def __init__(self, byteOrder):
    '''Creates a new WkbWriter.

    @param byteOrder Whether to write big-endian or little-endian data.
    '''

    if byteOrder not in ('big', 'little'):
      msg = "Invalid byte order: " + str(byteOrder)
      raise ValueError(msg)

    if byteOrder == 'big':
      self.byteOrder = '>' # Big endian
    else:
      self.byteOrder = '<' # Little endian

  #-----------------------------------------------------------------
  def getByteOrderData(self):
    '''Gets the integer code to write in the WKB data for this writer's order.
    '''

    wkbBig = 0
    wkbLittle = 1

    byteOrderData = wkbBig
    if self.byteOrder == '<':
      byteOrderData = wkbLittle

    return byteOrderData

  #-----------------------------------------------------------------
  def write(self, obj):
    '''Writes the object and returns the binary data.

    @param obj a python object.

    @return A string of hex-encoded binary data.
    '''

    data = b''

    # WKB types.

    wkbPoint = 1
    wkbLineString = 2
    wkbPolygon = 3
    wkbMultiPolygon = 6
    wkbPointz = 1001

    # Write the byte order.

    data += self.writeByte(self.getByteOrderData())

    # Write the object.

    if isinstance(obj, Point):
      data += self.writeInt(wkbPoint)
      data += self.writePoint(obj)
    elif isinstance(obj, Point3d):
      data += self.writeInt(wkbPointz)
      data += self.writePoint3d(obj)
    elif isinstance(obj, LineString):
      data += self.writeInt(wkbLineString)
      data += self.writePolygonRing(obj)
    elif isinstance(obj, Polygon):
      data += self.writeInt(wkbPolygon)
      data += self.writePolygon(obj)
    elif isinstance(obj, MultiPolygon):
      data += self.writeInt(wkbMultiPolygon)
      data += self.writeMultiPolygon(obj)
    else:
      msg = "Unsupported geometry type: " + str(obj.__class__)
      raise TypeError(msg)

    hexData = binascii.hexlify(data)

    # For some reason, hexlify() returns bytes rather than a string.
    hexStr = hexData.decode('utf-8')

    return hexStr

  #-----------------------------------------------------------------
  def writePoint(self, point: Point):
    '''Returns the binary data encoding for an object.

    @param point The Point object to encode.

    @return The binary data.
    '''

    fmt = self.byteOrder + "dd"

    # For Python 2, need to convert from unicode to string.
    fmt = str(fmt)

    return struct.pack(fmt, point.x, point.y)

  #-----------------------------------------------------------------
  def writePoint3d(self, point: Point3d):
    '''Returns the binary data encoding for an object.

    @param point The Point3d object to encode.

    @return The binary data.
    '''

    fmt = self.byteOrder + "ddd"

    fmt = str(fmt)

    return struct.pack(fmt, point.x, point.y, point.z)

  #-----------------------------------------------------------------
  def writePolygonRing(self, polygonRing):
    '''Returns the binary data encoding for an object.

    @param polygonRing The PolygonRing object to encode.

    @return The binary data.
    '''

    data = b''

    numPoints = len(polygonRing.points)
    data += self.writeInt(numPoints)

    for point in polygonRing.points:
      data += self.writePoint(point)

    return data

  #-----------------------------------------------------------------
  def writePolygon(self, polygon):
    '''Returns the binary data encoding for an object.

    @param polygon The Polygon object to encode.

    @return The binary data.
    '''

    data = b''

    numRings = len(polygon.rings)
    data += self.writeInt(numRings)

    for ring in polygon.rings:
      data += self.writePolygonRing(ring)

    return data

  #-----------------------------------------------------------------
  def writeMultiPolygon(self, multiPolygon):
    '''Returns the binary data encoding for an object.

    @param multiPolygon The MultiPolygon object to encode.

    @return The binary data.
    '''

    data = b''

    numPolygons = len(multiPolygon.polygons)
    data += self.writeInt(numPolygons)

    for polygon in multiPolygon.polygons:

      # For some reason, each Polygon starts with a byte order and
      # type code.
      data += self.writeByte(self.getByteOrderData())
      wkbPolygon = 3
      data += self.writeInt(wkbPolygon)

      data += self.writePolygon(polygon)

    return data

  #-----------------------------------------------------------------
  def writeInt(self, value):
    '''Returns the binary data encoding for an int.

    @param value The int to encode.

    @return The binary data.
    '''

    fmt = self.byteOrder + "I"

    # For Python 2, need to convert from unicode to string.
    fmt = str(fmt)

    return struct.pack(fmt, value)

  #-----------------------------------------------------------------
  def writeByte(self, value):
    '''Returns the binary data encoding for a 1-byte integer.

    @param value The value to encode.

    @return The binary data.
    '''

    fmt = self.byteOrder + "B"

    # For Python 2, need to convert from unicode to string.
    fmt = str(fmt)

    return struct.pack(fmt, value)
