#-------------------------------------------------------------------
#  WkbReader.py
#
#  The WkbReader class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WkbReader class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .GeoShape import GeoShape
from .LineString import LineString
from .Point import Point
from .Point3d import Point3d
from .PolygonRing import PolygonRing
from .Polygon import Polygon
from .MultiPolygon import MultiPolygon
import binascii
import struct
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WkbReader:
  '''Reads Well-Known Binary data into a python object.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WkbReader.
    '''

    pass

  #-----------------------------------------------------------------
  def read(self, dataStr: str) -> GeoShape:
    '''Reads the binary data and returns an object.

    @param dataStr A string of hex-encoded binary data.

    @return a python object.
    '''

    data: bytes = binascii.unhexlify(dataStr)
    dataIndex: int = 0

    if len(data) < 5:
      msg = "WTB data too short.  Expected at least 5 bytes. Data: " + str(data)
      raise ValueError(msg)

    # Read the byte order.

    byteOrderData, dataIndex = self.readByte(data, dataIndex)
    if byteOrderData == 0:
      byteOrder = '>' # Big endian
    elif byteOrderData == 1:
      byteOrder = '<' # Little endian
    else:
      msg = "Invalid byte order: " + str(byteOrderData)
      raise ValueError(msg)

    # Read the type.

    wkbPoint = 1
    wkbLineString = 2
    wkbPolygon = 3
    wkbMultiPolygon = 6
    wkbPointz = 1001

    geoType, dataIndex = self.readInt(byteOrder, data, dataIndex)

    if geoType == wkbPoint:
      point, dataIndex = self.readPoint(byteOrder, data, dataIndex)
      return point
    elif geoType == wkbPointz:
      point3d, dataIndex = self.readPoint3d(byteOrder, data, dataIndex)
      return point3d
    elif geoType == wkbLineString:
      ring, dataIndex = self.readPolygonRing(byteOrder, data, dataIndex)
      return LineString(ring.points)
    elif geoType == wkbPolygon:
      polygon, dataIndex = self.readPolygon(byteOrder, data, dataIndex)
      return polygon
    elif geoType == wkbMultiPolygon:
      mPolygon, dataIndex = self.readMultiPolygon(byteOrder, data, dataIndex)
      return mPolygon
    else:
      msg = ("Unsupported geometry type: " + str(geoType) +
             " Data: " + str(dataStr))
      raise ValueError(msg)

  #-----------------------------------------------------------------
  def readPoint(self,
                byteOrder: str,
                data: bytes,
                dataIndex: int) -> Tuple[Point, int]:
    '''Reads the binary data and returns a Point object.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (Point object, new dataIndex)
    '''

    startIndex = dataIndex
    endIndex = dataIndex + 16

    fmt: str = byteOrder + "dd"

    x, y = struct.unpack(fmt, data[startIndex:endIndex])

    return (Point(x, y), endIndex)

  #-----------------------------------------------------------------
  def readPoint3d(self,
                byteOrder: str,
                data: bytes,
                dataIndex: int) -> Tuple[Point3d, int]:
    '''Reads the binary data and returns a Point3d object.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (Point3d object, new dataIndex)
    '''

    startIndex = dataIndex
    endIndex = dataIndex + 24

    fmt: str = byteOrder + "ddd"

    x, y, z = struct.unpack(fmt, data[startIndex:endIndex])

    return (Point3d(x, y, z), endIndex)

  #-----------------------------------------------------------------
  def readPolygonRing(self,
                      byteOrder: str,
                      data: bytes,
                      dataIndex: int) -> Tuple[PolygonRing, int]:
    '''Reads the binary data and returns a PolygonRing object.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (PolygonRing object, new dataIndex)
    '''

    (numPoints, dataIndex) = self.readInt(byteOrder, data, dataIndex)

    points = []

    for _ in range(numPoints):
      point, dataIndex = self.readPoint(byteOrder, data, dataIndex)
      points.append(point)

    return (PolygonRing(points), dataIndex)

  #-----------------------------------------------------------------
  def readPolygon(self,
                  byteOrder: str,
                  data: bytes,
                  dataIndex: int) -> Tuple[Polygon, int]:
    '''Reads the binary data and returns a Polygon object.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (Polygon object, new dataIndex)
    '''

    (numRings, dataIndex) = self.readInt(byteOrder, data, dataIndex)

    rings = []

    for _ in range(numRings):
      ring, dataIndex = self.readPolygonRing(byteOrder, data, dataIndex)
      rings.append(ring)

    return (Polygon(rings), dataIndex)

  #-----------------------------------------------------------------
  def readMultiPolygon(self,
                  byteOrder: str,
                  data: bytes,
                  dataIndex: int) -> Tuple[MultiPolygon, int]:
    '''Reads the binary data and returns a MultiPolygon object.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (MultiPolygon object, new dataIndex)
    '''

    (numPolygons, dataIndex) = self.readInt(byteOrder, data, dataIndex)

    polygons = []

    for _ in range(numPolygons):

      # For some reason, each Polygon starts with a byte order and
      # type code.  Read and ignore these.
      ignoredValue, dataIndex = self.readByte(data, dataIndex)
      ignoredValue, dataIndex = self.readInt(byteOrder, data, dataIndex)

      polygon, dataIndex = self.readPolygon(byteOrder, data, dataIndex)
      polygons.append(polygon)

    return (MultiPolygon(polygons), dataIndex)

  #-----------------------------------------------------------------
  def readInt(self,
              byteOrder: str,
              data: bytes,
              dataIndex: int) -> Tuple[int, int]:
    '''Reads the binary data and returns an int.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (int, new dataIndex)
    '''

    return self.readValue(byteOrder, 'I', 4, data, dataIndex)

  #-----------------------------------------------------------------
  def readByte(self, data: bytes, dataIndex: int) -> Tuple[int, int]:
    '''Reads the binary data and returns an int.

    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (int, new dataIndex)
    '''

    # Don't need a byte order since we're reading a single byte.
    byteOrder = ''

    return self.readValue(byteOrder, 'B', 1, data, dataIndex)

  #-----------------------------------------------------------------
  def readValue(self,
                byteOrder: str,
                code: str,
                size: int,
                data: bytes,
                dataIndex: int) -> Tuple[int, int]:
    '''Reads the binary data and returns a single value.

    @param byteOrder The struct.unpack byte order format string.
    @param code The struct.unpack type format string.
    @param size The number of bytes to read.
    @param data A string of hex-encoded binary data.
    @param dataIndex The current read index in the data.

    @return a tuple: (int, new dataIndex)
    '''

    startIndex = dataIndex
    endIndex = startIndex + size

    fmt = byteOrder + code
    dataSlice = data[startIndex:endIndex]

    (value, ) = struct.unpack(fmt, dataSlice)

    return (value, endIndex)

  #-----------------------------------------------------------------
  def dump(self, data: bytes, dataIndex: int) -> str:
    return data[dataIndex:dataIndex + 20].hex()
