#-------------------------------------------------------------------
#  WkbRasterWriter.py
#
#  The WkbRasterWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WkbRasterWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import RasterBand
import binascii
import struct
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WkbRasterWriter:
  '''Writes a python object into Well-Known Binary data.
  '''

  #-----------------------------------------------------------------
  def __init__(self, byteOrder):
    '''Creates a new WkbRasterWriter.

    @param byteOrder Whether to write big-endian or little-endian data.
    '''

    if byteOrder not in ('big', 'little'):
      msg = "Invalid byte order: " + str(byteOrder)
      raise ValueError(msg)

    if byteOrder == 'big':
      self.byteOrder = '>' # Big endian
    else:
      self.byteOrder = '<' # Little endian

  #-----------------------------------------------------------------
  def getByteOrderData(self):
    '''Gets the integer code to write in the WKB data for this writer's order.
    '''

    wkbBig = 0
    wkbLittle = 1

    byteOrderData = wkbBig
    if self.byteOrder == '<':
      byteOrderData = wkbLittle

    return byteOrderData

  #-----------------------------------------------------------------
  def write(self, raster):
    '''Writes the object and returns the binary data.

    @param raster The raster to encode into WKB.

    @return A string of hex-encoded binary data.
    '''

    raster.validate()

    data = b''

    # Write the byte order.

    data += self.writeByte(self.getByteOrderData())

    # Write the version.

    version = 0
    data += self.writeValue('uint16', version)

    # Write the raster data.

    data += self.writeValue('uint16', len(raster.bands))
    data += self.writeValue('float64', raster.scaleX)
    data += self.writeValue('float64', raster.scaleY)
    data += self.writeValue('float64', raster.upperLeftX)
    data += self.writeValue('float64', raster.upperLeftY)
    data += self.writeValue('float64', raster.skewX)
    data += self.writeValue('float64', raster.skewY)
    data += self.writeValue('int32', raster.srid)
    data += self.writeValue('uint16', raster.width)
    data += self.writeValue('uint16', raster.height)

    # Write the raster bands.

    for band in raster.bands:

      data += self.writeBand(band)

    hexData = binascii.hexlify(data)

    # For some reason, hexlify() returns bytes rather than a string.
    hexStr = hexData.decode('utf-8')

    return hexStr

  #-----------------------------------------------------------------
  def writeBand(self, band):
    '''Write a RasterBand into binary data.

    @param band the RasterBand object to write.

    @return Binary data, the encoded value.
    '''

    # pylint: disable=C0200

    data = b''

    isOffline = False
    hasNoData = band.noData is not None
    isAllNoData = band.isAllNoData
    pixelTypeInt = band.pixelType

    packedData = 0
    packedData = packedData | (isOffline << 7)
    packedData = packedData | (hasNoData << 6)
    packedData = packedData | (isAllNoData << 5)
    packedData = packedData | pixelTypeInt

    pixelType = RasterBand.pixelTypeCodes[pixelTypeInt]

    data += self.writeByte(packedData)

    noData = band.noData
    if noData is None:
      noData = 0

    data += self.writeValue(pixelType, noData)

    values = band.values

    flatValues = []
    for row in values:
      flatValues.extend(row)

    # Replace None with the noData value.
    if hasNoData:
      for i in range(len(flatValues)):
        if flatValues[i] is None:
          flatValues[i] = noData

    data += self.writeValue(pixelType, flatValues)

    return data

  #-----------------------------------------------------------------
  def writeByte(self, value):
    '''Writes a python value into a byte of binary data.

    @param value The value to write.

    @return Binary data, the encoded value.
    '''

    return self.writeValueCode('B', value)

  #-----------------------------------------------------------------
  def writeValue(self, valueType, value):
    '''Writes a python value into binary data.

    @param valueType A type string such as 'uint16'.
    @param value The value to write.

    @return Binary data, the encoded value.
    '''

    # Map of type names to struct format codes.
    valueTypes = {
      'bool' : 'B',
      'int2' : 'B',
      'int4' : 'B',
      'int8' : 'b',
      'uint8' : 'B',
      'int16' : 'h',
      'uint16' : 'H',
      'int32' : 'i',
      'uint32' : 'I',
      'float32' : 'f',
      'float64' : 'd',
     }

    if valueType not in valueTypes:
      raise ValueError('Invalid valueType: %s' % valueType)

    code = valueTypes[valueType]

    return self.writeValueCode(code, value)

  #-----------------------------------------------------------------
  def writeValueCode(self, code, value):
    '''Writes a python value into binary data.

    @param code The struct.pack type format string.
    @param value The value to write.

    @return Binary data, the encoded value.
    '''

    numValues = 1
    if isinstance(value, (list, tuple)):
      numValues = len(value)

    fmt = self.byteOrder + (code * numValues)

    # For Python 2, need to convert from unicode to string.
    fmt = str(fmt)

    if numValues == 1:
      packedData = struct.pack(fmt, value)
    else:
      packedData = struct.pack(fmt, *value)

    return packedData

  #-----------------------------------------------------------------
  def dump(self, data, numBytes=8):
    return data[-numBytes:].encode('hex')
