#-------------------------------------------------------------------
#  RasterBand.py
#
#  The RasterBand class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the RasterBand class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class RasterBand:
  '''A single band of data in a raster.
  '''

  #-----------------------------------------------------------------
  # Map of pixel type integers to strings.
  #
  # The WKB raster docs say 9 and 10 are float 32 and 64, but the software
  # seems to actually use 10 and 11.
  #
  pixelTypeCodes = {
    0 : 'bool',
    1 : 'int2',
    2 : 'int4',
    3 : 'int8',
    4 : 'uint8',
    5 : 'int16',
    6 : 'uint16',
    7 : 'int32',
    8 : 'uint32',
    10 : 'float32',
    11 : 'float64',
  }

  #-----------------------------------------------------------------
  def __init__(self, pixelType, noData, isAllNoData, values):
    '''Creates a new RasterBand.
    '''

    # The pixel data type of this band.
    #
    # 0: 1-bit boolean
    # 1: 2-bit unsigned integer
    # 2: 4-bit unsigned integer
    # 3: 8-bit signed integer
    # 4: 8-bit unsigned integer
    # 5: 16-bit signed integer
    # 6: 16-bit unsigned signed integer
    # 7: 32-bit signed integer
    # 8: 32-bit unsigned signed integer
    # 10: 32-bit float
    # 11: 64-bit float
    #
    self.pixelType = pixelType

    # The NODATA value for this band.
    #
    # If None, this band does not have a NODATA value.
    self.noData = noData

    # If true, all value items should be treated as NODATA, regardless
    # of their actual value.
    self.isAllNoData = isAllNoData

    # A 2-D array of data.
    #
    # Any NODATA values will be None in this array.
    #
    self.values = values

  #-----------------------------------------------------------------
  def __repr__(self):
    '''Formats this object into a string.
    '''

    attrs = ['pixelType', 'noData', 'isAllNoData']
    return StringUtil.formatRepr(self, attrs)
