#-------------------------------------------------------------------
#  Raster.py
#
#  The Raster class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Raster class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .RasterBand import RasterBand
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class Raster:
  '''A PostGIS raster object.
  '''

  #-----------------------------------------------------------------
  def __init__(self, width, height, upperLeftX, upperLeftY, scaleX, scaleY,
               skewX, skewY, srid, bands):
    '''Creates a new Raster.
    '''

    # The number of pixel columns.
    self.width = width

    # The number of pixel rows.
    self.height = height

    # The x coordinate in geographical units of the upper-left corner.
    self.upperLeftX = upperLeftX

    # The y coordinate in geographical units of the upper-left corner.
    self.upperLeftY = upperLeftY

    # The width of a single pixel in geographical units.
    self.scaleX = scaleX

    # The height of a single pixel in geographical units.
    self.scaleY = scaleY

    # Rotation about the Y-axis.
    self.skewX = skewX

    # Rotation about the X-axis.
    self.skewY = skewY

    # The spacial reference ID.
    self.srid = srid

    # The bands of this raster.
    self.bands = bands

  #-----------------------------------------------------------------
  @staticmethod
  def create(upperLeftX, upperLeftY, scaleX, scaleY, values):
    '''Creates a 1-band raster in 4326.
    '''

    width = len(values[0])
    height = len(values)

    band = RasterBand(10, None, False, values)
    raster = Raster(width, height, upperLeftX, upperLeftY, scaleX, scaleY,
                    0, 0, 4326, [band])

    return raster

  #-----------------------------------------------------------------
  def createCopy(self):
    '''Creates a new copy of this object.
    '''

    # Don't copy the bands.
    bands = []

    return Raster(self.width,
                  self.height,
                  self.upperLeftX,
                  self.upperLeftY,
                  self.scaleX,
                  self.scaleY,
                  self.skewX,
                  self.skewY,
                  self.srid,
                  bands)

  #-----------------------------------------------------------------
  def validate(self):
    '''Validates that this raster is self-consistent.
    '''

    for bandNum, band in enumerate(self.bands):

      values = band.values

      if len(values) != self.height:
        msg = 'Error in band %s.  Band has %s rows, but this raster has '
        msg += 'a height of %s.'
        msg = msg % (bandNum, len(values), self.height)
        raise ValueError(msg)

      for rowNum, row in enumerate(values):

        if len(row) != self.width:
          msg = 'Error in band %s row %s.  Row has %s columns, but this raster '
          msg += 'has a width of %s.'
          msg = msg % (bandNum, rowNum, len(row), self.width)
          raise ValueError(msg)

  #-----------------------------------------------------------------
  def __repr__(self):
    '''Formats this object into a string.
    '''

    attrs = ['width', 'height', 'upperLeftX', 'upperLeftY', 'scaleX', 'scaleY',
             'skewX', 'skewY', 'srid', 'bands']
    return StringUtil.formatRepr(self, attrs)
