#-------------------------------------------------------------------
#  Polygon.py
#
#  The Polygon class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Polygon class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .GeoShape import GeoShape
from .PolygonRing import PolygonRing
from .Point import Point
from ai.axe.util import StringUtil
from typing import Callable
from typing import List
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class Polygon(GeoShape):
  '''A WKT Polygon, made of up a list of rings.

  The first ring is the exterior of the polygon, and each
  following ring is a 'hole' in the polygon.
  '''

  #-----------------------------------------------------------------
  def __init__(self, rings: List[PolygonRing]) -> None:
    '''Creates a new Polygon.

    @param rings A list of PolygonRing objects that defines the border and
                 holes of this polygon.
    '''

    GeoShape.__init__(self)

    assert isinstance(rings, (list, tuple)), rings
    for i, ring in enumerate(rings):
      assert isinstance(ring, PolygonRing), (i, ring)

    # A list of PolygonRing objects that defines the border and
    # holes of this polygon.
    self.rings = rings

  #-----------------------------------------------------------------
  def createCopy(self, modifyFunction: Callable[[Point], Point] = None):
    '''Creates a new copy of this polygon.

    @param modifyMethod An optional function to be applied to each Point.

    @return A new copy of this Polygon.
    '''

    # If no modify function was passed in, use a function that
    # just returns its argument unchanged.
    if not modifyFunction:
      modifyFunction = lambda x: x

    newRings = []

    for ring in self.rings:
      newPoints = []
      for oldPoint in ring.points:
        newPoints.append(modifyFunction(oldPoint))
      newRings.append(PolygonRing(newPoints))

    return Polygon(newRings)

  #-----------------------------------------------------------------
  @staticmethod
  def createRectangle(upperLeftXy: Tuple[float, float],
                      lowerRightXy: Tuple[float, float]) -> 'Polygon':
    '''Creates a new polygon that is a rectangle.

    @param upperLeftXy The upper left corner of the created rectangle.
    @param lowerRightXy The lower right corner of the created rectangle.

    @return A newly created Polygon.
    '''

    x1, y2 = upperLeftXy
    x2, y1 = lowerRightXy

    points = [Point(x1, y2),
              Point(x2, y2),
              Point(x2, y1),
              Point(x1, y1),
              Point(x1, y2)]

    rings = [PolygonRing(points)]
    return Polygon(rings)

  #-----------------------------------------------------------------
  @staticmethod
  def createFromPoints(xyList):
    '''Creates a new polygon from a list of (x, y) tuples.

    @param xyList A list of (x, y) tuples.

    @return A newly created Polygon.
    '''

    points = []

    for xy in xyList:
      points.append(Point(*xy))

    rings = [PolygonRing(points)]
    return Polygon(rings)

  #-----------------------------------------------------------------
  def add(self, point: Point) -> 'Polygon':
    '''Adds the given point to all points in this polygon.

    @return A new copy of this polygon with the points added.
    '''

    addFunc = lambda oldPoint: oldPoint.add(point)

    return self.createCopy(addFunc)

  #-----------------------------------------------------------------
  def subtract(self, point: Point) -> 'Polygon':
    '''Subtracts the given point from all points in this polygon.

    @return A new copy of this polygon with the points subtracted.
    '''

    subtractFunc = lambda oldPoint: oldPoint.subtract(point)

    return self.createCopy(subtractFunc)

  #-----------------------------------------------------------------
  def multiply(self, point: Point) -> 'Polygon':
    '''Multiplies the given point to all points in this polygon.

    @return A new copy of this polygon with the points multiplied.
    '''

    multiplyFunc = lambda oldPoint: oldPoint.multiply(point)

    return self.createCopy(multiplyFunc)

  #-----------------------------------------------------------------
  def divide(self, point: Point) -> 'Polygon':
    '''Divides all points in this polygon by the given point.

    @return A new copy of this polygon with the points divided.
    '''

    divideFunc = lambda oldPoint: oldPoint.divide(point)

    return self.createCopy(divideFunc)

  #-----------------------------------------------------------------
  def __eq__(self, other):
    '''Support the == operator.
    '''

    if not hasattr(other, 'rings'):
      return False

    return self.rings == other.rings

  #-----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Formats this object into a string.
    '''

    attrs = ['rings']
    return StringUtil.formatRepr(self, attrs)
