#-------------------------------------------------------------------
#  Point3d.py
#
#  The Point3d class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Point3d class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.util import StringUtil
from .GeoShape import GeoShape
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
@ClassJsonClass([ClassJsonField('x', float),
                 ClassJsonField('y', float),
                 ClassJsonField('z', float)])
class Point3d(GeoShape):
  '''An (x, y, z) point.
  '''

  #-----------------------------------------------------------------
  def __init__(self, x: float, y: float, z: float) -> None:
    '''Creates a new Point3d.
    '''

    GeoShape.__init__(self)

    # The X coordinate of this point.
    self.x: float = x

    # The Y coordinate of this point.
    self.y: float = y

    # The Z coordinate of this point.
    self.z: float = z

  #-----------------------------------------------------------------
  def __setattr__(self, name: str, value: object) -> None:
    '''Implement __setattr__ to disallow changing values.
    '''

    if hasattr(self, name):
      msg = "Attempted to change a value, but Point3d is immutable.  %s=%s"
      msg = msg % (name, value)
      raise TypeError(msg)

    else:
      object.__setattr__(self, name, value)

  #-----------------------------------------------------------------
  def __eq__(self, other) -> bool:
    '''Support the == operator.
    '''

    return (hasattr(other, 'x') and
            hasattr(other, 'y') and
            hasattr(other, 'z') and
            self.x == other.x and
            self.y == other.y and
            self.z == other.z)

  #-----------------------------------------------------------------
  def isCloseTo(self, other: 'Point3d', tolerance: float = 0.00001):
    '''Returns true if the other point is close to this one.

    @param other The point to compare this one to.

    @param tolerance The two points must be within the specified
                     absolute tolerance.  The default tolerance is
                     0.00001, which requires the points to be within
                     one meter of each other (at the equator) if these
                     points are degrees.

    @return True or False.
    '''

    return (abs(self.x - other.x) < tolerance and
            abs(self.y - other.y) < tolerance and
            abs(self.z - other.z) < tolerance)

  #-----------------------------------------------------------------
  def add(self, point: 'Point3d') -> 'Point3d':
    '''Adds the specified point to this one.

    @return A new copy of this object with the point added.
    '''

    newX = self.x + point.x
    newY = self.y + point.y
    newZ = self.z + point.z

    return Point3d(newX, newY, newZ)

  #-----------------------------------------------------------------
  def subtract(self, point: 'Point3d') -> 'Point3d':
    '''Subtract the specified point from this one.

    @return A new copy of this object with the point subtracted.
    '''

    newX = self.x - point.x
    newY = self.y - point.y
    newZ = self.z - point.z

    return Point3d(newX, newY, newZ)

  #-----------------------------------------------------------------
  def multiply(self, point: 'Point3d') -> 'Point3d':
    '''Multiplies the specified point to this one.

    @return A new copy of this object with the point multiplied.
    '''

    newX = self.x * point.x
    newY = self.y * point.y
    newZ = self.z * point.z

    return Point3d(newX, newY, newZ)

  #-----------------------------------------------------------------
  def divide(self, point: 'Point3d') -> 'Point3d':
    '''Divides this point by the specified point.

    @return A new copy of this object with the point divided.
    '''

    newX = self.x / point.x
    newY = self.y / point.y
    newZ = self.z / point.z

    return Point3d(newX, newY, newZ)

  #-----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Formats this object into a string.
    '''

    attrs = ['x', 'y', 'z']
    return StringUtil.formatRepr(self, attrs)
