#-------------------------------------------------------------------
#  Point.py
#
#  The Point class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Point class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.util import StringUtil
from .GeoShape import GeoShape
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
@ClassJsonClass([ClassJsonField('x', float),
                 ClassJsonField('y', float)])
class Point(GeoShape):
  '''An (x, y) point.
  '''

  #-----------------------------------------------------------------
  def __init__(self, x: float, y: float) -> None:
    '''Creates a new Point.
    '''

    GeoShape.__init__(self)

    # The X coordinate of this point.
    self.x: float = x

    # The Y coordinate of this point.
    self.y: float = y

  #-----------------------------------------------------------------
  def __setattr__(self, name: str, value: object) -> None:
    '''Implement __setattr__ to disallow changing values.
    '''

    if hasattr(self, name):
      msg = "Attempted to change a value, but Point is immutable.  %s=%s"
      msg = msg % (name, value)
      raise TypeError(msg)

    else:
      object.__setattr__(self, name, value)

  #-----------------------------------------------------------------
  def __eq__(self, other) -> bool:
    '''Support the == operator.
    '''

    return (hasattr(other, 'x') and hasattr(other, 'y') and
            self.x == other.x and self.y == other.y)

  #-----------------------------------------------------------------
  def isCloseTo(self, other: 'Point', tolerance: float = 0.00001):
    '''Returns true if the other point is close to this one.

    @param other The point to compare this one to.

    @param tolerance The two points must be within the specified
                     absolute tolerance.  The default tolerance is
                     0.00001, which requires the points to be within
                     one meter of each other (at the equator) if these
                     points are degrees.

    @return True or False.
    '''

    return (abs(self.x - other.x) < tolerance and
            abs(self.y - other.y) < tolerance)

  #-----------------------------------------------------------------
  def add(self, point: 'Point') -> 'Point':
    '''Adds the specified point to this one.

    @return A new copy of this object with the point added.
    '''

    newX = self.x + point.x
    newY = self.y + point.y

    return Point(newX, newY)

  #-----------------------------------------------------------------
  def subtract(self, point: 'Point') -> 'Point':
    '''Subtract the specified point from this one.

    @return A new copy of this object with the point subtracted.
    '''

    newX = self.x - point.x
    newY = self.y - point.y

    return Point(newX, newY)

  #-----------------------------------------------------------------
  def multiply(self, point: 'Point') -> 'Point':
    '''Multiplies the specified point to this one.

    @return A new copy of this object with the point multiplied.
    '''

    newX = self.x * point.x
    newY = self.y * point.y

    return Point(newX, newY)

  #-----------------------------------------------------------------
  def divide(self, point: 'Point') -> 'Point':
    '''Divides this point by the specified point.

    @return A new copy of this object with the point divided.
    '''

    newX = self.x / point.x
    newY = self.y / point.y

    return Point(newX, newY)

  #-----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Formats this object into a string.
    '''

    attrs = ['x', 'y']
    return StringUtil.formatRepr(self, attrs)
