#-------------------------------------------------------------------
#  EsriPolygon.py
#
#  The EsriPolygon class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the EsriPolygon class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.util import StringUtil
from .EsriSpatialReference import EsriSpatialReference
from .MultiPolygon import MultiPolygon
from .Polygon import Polygon
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
@ClassJsonClass([ClassJsonField('rings', 'any'),
                 ClassJsonField('spatialReference', EsriSpatialReference)],
                encodeOnly=True)
class EsriPolygon:
  '''A Python equivalent to the ESRI Javascript esri.geometry.Polygon class.

  This can be both a polygon and a multi-polygon.  ESRI looks at each
  ring and decides whether it's an exterior ring or a "hole" ring.
  '''

  #-----------------------------------------------------------------
  def __init__(self, rings, wkid):
    '''Creates a new EsriPolygon.
    '''

    # List of rings.  A ring is list of points.  A point is a list of 2 numbers.
    self.rings = rings

    # The spatial reference of the polygon.
    self.spatialReference = EsriSpatialReference(wkid)

  #-----------------------------------------------------------------
  @staticmethod
  def fromPolygon(polygon):
    '''Creates an EsriPolygon from a Polygon or MultiPolygon object.

    @param polygon A Polygon or MultiPolygon object.

    @return An EsriPolygon object.
    '''

    if not isinstance(polygon, (Polygon, MultiPolygon)):

      msg = "Expected gmc.util.Polygon or MultiPolygon, got: "
      msg += str(polygon.__class__)
      raise TypeError(msg)

    polygons = []

    if isinstance(polygon, Polygon):
      polygons.append(polygon)
    else:
      polygons.extend(polygon.polygons)

    esriRings = []

    for srcPolygon in polygons:
      for srcRing in srcPolygon.rings:

        esriRing = [(point.x, point.y) for point in srcRing.points]
        esriRings.append(esriRing)

    esriPolygon = EsriPolygon(esriRings, 4326)

    return esriPolygon

  #----------------------------------------------------------------
  def __repr__(self):
    '''Returns a string representation of this object
    '''
    attrs = ["rings", "spatialReference"]

    return StringUtil.formatRepr(self, attrs)
