#-------------------------------------------------------------------
#  SyncTools.py
#
#  The SyncTools module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The 'syncTools' script implementation.

To use this in a dev environment, make a script called 'syncTools'
whose implementation is:

  SyncTools.execute(appName='foo',
                    storeDir='https://s3.amazonaws.com/foo/developers/')
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import hashlib
import os.path
import platform as platformModule
import re
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
usage = '''\
syncTools

Downloads and unpacks the tools for the current version of your repo.

  -f   Force download.  If this argument is supplied, the tools will
                        re-downloaded even if they already exist.
'''

#-------------------------------------------------------------------
def execute(appName: str, storeDir: str) -> None:
  '''Executes the 'syncTools' command.

  @param appName The name of the application, such as 'foo'.
  @param storeDir The URL of a directory that contains tool tarballs.
                  For example:
                  https://s3.amazonaws.com/ai-seeds/Ram/developers/
  '''

  assert appName and appName[0].islower(), ('Bad appName: %s' % appName)

  appNameAllCaps = camelCaseToUnderscores(appName).upper()

  appTools = appName + "Tools"

  # Parse command-line arguments.

  args = sys.argv[1:]

  if '-h' in args or '--help' in args:
    print(usage)
    sys.exit(1)

  forceDownload = False

  while args:

    arg = args.pop(0)

    if arg == '-f':
      forceDownload = True

    else:
      print("Unknown argument: " + arg)
      sys.exit(1)

  home = os.environ[appNameAllCaps + '_HOME']

  versionFile = home + '/toolsVersion.txt'
  text = open(versionFile).read().strip()
  line3 = text.split('\n')[2]

  version, linuxHash, darwinHash = line3.split()

  hashes = {}
  hashes['linux'] = linuxHash
  hashes['darwin'] = darwinHash

  platform = platformModule.system().lower()

  assert platform in ('linux', 'darwin'), platform

  fileName = appTools + '-' + version + '-' + platform + '.tar.gz'

  os.chdir(home + '/tools')

  if forceDownload and os.path.exists(fileName):

    # Delete the old file.
    print('Forcing download because -f flag was passed in.')
    os.unlink(fileName)

  if os.path.exists(fileName):
    print('File already exists, checking checksum....')

    if md5sum(fileName) == hashes[platform]:
      print('Existing file looks good, skipping download.')

    else:
      # Delete the bad file.
      os.unlink(fileName)
      print('Existing file is bad. Re-downloading.')

  if not os.path.exists(fileName):
    cmd = 'curl ' + storeDir + fileName + ' -o ' + fileName
    executeCmd(cmd)

  if md5sum(fileName) != hashes[platform]:
    print('ERROR')
    print('ERROR')
    print('ERROR')
    print('ERROR')
    print('ERROR')
    print('ERROR Download appears to have failed.  Aborting.')
    sys.exit(1)

  executeCmd('rm -rf ' + appTools)
  executeCmd('tar zxf ' + fileName)

#-------------------------------------------------------------------
def executeCmd(cmd: str) -> None:
  """Execute the specified command.
  """

  print(cmd)
  status = os.system(cmd)
  if status != 0:
    msg = ("Error in script 'syncTools'.  The commmand '" + cmd +
           "' had an exit status of " + str(status) + ".")
    raise RuntimeError(msg)

#-------------------------------------------------------------------
def md5sum(fileName: str) -> str:
  '''Returns an md5 hash for an object with read() method.'''

  fobj = open(fileName, 'rb')

  m = hashlib.md5()

  while True:
    d = fobj.read(8096)
    if not d:
      break
    m.update(d)

  return m.hexdigest()

#-------------------------------------------------------------------
def camelCaseToUnderscores(text: str) -> str:
  '''Returns a copy of the text with camel case converted to underscores.
  '''

  camelCaseRe1 = re.compile('(.)([A-Z][a-z]+)')
  camelCaseRe2 = re.compile('([a-z0-9])([A-Z])')

  text = camelCaseRe1.sub(r'\1_\2', text)
  text = camelCaseRe2.sub(r'\1_\2', text)
  text = text.lower()
  return text
