#-------------------------------------------------------------------
#  SetUpDevArea.py
#
#  The SetUpDevArea module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

"""Sets up a webapp development area.

Copies all files neccessary to make a source tree
able to function as a web app.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
from glob import glob
from typing import List
from typing import Optional
import sys
import os.path
import re
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
usage = """\
setUpDevArea [-v]

Passing "-v" turns on verbose output.
"""

#-------------------------------------------------------------------
# Global variable.  True if verbose output is on.
verbose = False


#-------------------------------------------------------------------
def execute(appName: str) -> None:
  '''Runs the command.

  @param appName The name of the application, such as 'ram' or 'cropDoctor'.
  '''

  appNameUpper = camelCaseToUnderscores(appName).upper()

  # Process the command line args.

  args = sys.argv[1:]

  if "-h" in args or "--help" in args:
    print(usage)
    sys.exit(0)

  if "-v" in args:

    # pylint: disable=W0603
    global verbose
    # pylint: enable=W0603

    verbose = True
    args.remove("-v")

  if len(args) > 1:
    print(usage)
    sys.exit(0)

  # Grab locations from environment variables.

  homeDir = os.environ[appNameUpper + "_HOME"]

  # Create the wepapp directories.

  createWebappDir("webapp", homeDir)

#-------------------------------------------------------------------
def createWebappDir(webappDirName: str, homeDir: str) -> None:
  """Creates a build/webapp directory.

  @param webappDirName The name of the directory to create under 'build'.
  @param webAppUrl The URL you want the webapp to have.
  """

  print('Setting up symlinks into build/' + webappDirName + ' directory...')

  destWebApp = os.path.join(homeDir, 'build', webappDirName)
  srcCoreWebApp = os.path.join(homeDir, 'src', 'webapp')

  # The RAM files.
  linkDontReplace(srcCoreWebApp, destWebApp)

#-------------------------------------------------------------------
def createParentDirs(dir: str) -> None:
  """Creates any missing parent dirs."""

  parent = os.path.split(dir)[0]
  if not os.path.exists(parent):
    os.makedirs(parent)

#-------------------------------------------------------------------
def linkTree(srcDir: str,
             destDir: str,
             srcFiles: List[str],
             ignoreFileList: List[str]) -> None:
  """Creates tree of a symlink."""

  for srcFile in srcFiles:
    fileName = srcFile[len(srcDir) + 1:]
    destFile = destDir + "/" + fileName

    ignore = False
    for ignoreFile in ignoreFileList:
      if re.search(ignoreFile, fileName):
        if verbose:
          print("NOT linking file '" + srcFile + "'.")
        ignore = True
        break
    if ignore:
      continue

    link(srcFile, destFile)

#-------------------------------------------------------------------
def link(src: str, dest: str) -> None:
  """Creates a symlink."""

  if os.path.lexists(dest):
    os.unlink(dest)

  if verbose:
    print("Linking '" + src + "' to '" + dest + "'.")
  createParentDirs(dest)
  os.symlink(src, dest)

#-------------------------------------------------------------------
def linkDontReplace(srcDir: str,
                    destDir: str,
                    ignoreFile: Optional[str] = None) -> None:
  """Links all files from the src to dest dir, but doesn't replace any.
  """

  srcFiles = glob(srcDir + '/*')

  for srcFile in srcFiles:
    fileName = srcFile[len(srcDir) + 1:]
    destFile = destDir + "/" + fileName

    if ignoreFile and re.match(ignoreFile, fileName):
      if verbose:
        print("NOT linking file '" + srcFile + "'.")
      continue

    if not os.path.exists(destFile):
      if verbose:
        print("Linking file '" + srcFile + "' to '" + destFile + "'.")
      createParentDirs(destFile)
      os.symlink(srcFile, destFile)

#-------------------------------------------------------------------
def camelCaseToUnderscores(text: str) -> str:
  '''Returns a copy of the text with camel case converted to underscores.
  '''

  camelCaseRe1 = re.compile('(.)([A-Z][a-z]+)')
  camelCaseRe2 = re.compile('([a-z0-9])([A-Z])')

  text = camelCaseRe1.sub(r'\1_\2', text)
  text = camelCaseRe2.sub(r'\1_\2', text)
  text = text.lower()
  return text
