#-------------------------------------------------------------------
#  GenBuildReport.py
#
#  The GenBuildReport module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Generate a report of what modules and tests are failing.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build import BuildSetup
from ai.axe.web.app import AppSetup
from collections import OrderedDict
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
import datetime
import os
import subprocess
import sys
from glob import glob
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
usage = """\
genBuildReport [browserTestServer]

Runs test and lint commands, and writes a report of what failed.

The browserTestServer parameter can be either 'local' (the default) or
'daily'.  This controls which server the tests will be run against.
"""

#-------------------------------------------------------------------
def execute() -> int:
  '''Executes this script.
  '''

  # Will be true only if this script is being run in a webapp environement
  # which contains a front end.
  hasFrontEnd = AppSetup.exists() and AppSetup.get().hasFrontEnd()

  args = sys.argv[1:]

  if '--help' in args or '-h' in args:
    print(usage)
    sys.exit(0)

  if len(args) > 1:
    print("Error: Too many args")
    sys.exit(1)

  browserTestServer = 'local'
  if args:
    arg = args.pop(0)
    if arg not in ('local', 'daily'):
      msg = 'Invalid browserTestServer argument. '
      msg += ' Must be "local" or "daily", but was: ' + str(arg)
      print(msg)
      sys.exit(1)
    browserTestServer = arg

  # Read the current build ID.

  cmd = "git describe --tags --long"
  out, err = executeSuccess(cmd)
  buildName = out.strip()

  tag, numCommits, commitName = buildName.rsplit('-', 2)

  # Suppress unused variable warning.
  err = err # pylint: disable=self-assigning-variable

  # Get rid of 'g' prefix.
  commitName = commitName[1:]

  nowStr = datetime.datetime.now().strftime("%a %b %d %I:%M %p")

  buildName = "Generated %s at git commit %s, which is %s commits after tag %s."
  buildName = buildName % (nowStr, commitName, numCommits, tag)

  # Dictionary of [className] = errorDict, where errorDict
  # is [testName] = errorMsg.
  errors: Dict[str, Dict[str, str]] = {}

  # Dictionary of [test name] = errorMsg.
  browserErrors: Optional[Dict[str, str]] = None

  runUnitTests(errors)
  runPylint(errors)

  if hasFrontEnd:
    runTslint(errors)
    browserErrors = runBrowserTests(browserTestServer)

  # Write the HTML report.

  writeReport(buildName, errors, browserErrors)

  # The return code is the number of errors.

  numBrowserErrors = len(browserErrors) if browserErrors else 0
  return len(errors) + numBrowserErrors

#-------------------------------------------------------------------
def runUnitTests(errors: Dict[str, Dict[str, str]]) -> None:
  '''Run the unit tests.

  @param errors Dictionary of [className] = errorDict, where errorDict
                is [testName] = errorMsg.
  '''

  # Read the list of unit tests to run.

  testedFile = 'build/unittest/tested.txt'
  if not os.path.exists(testedFile):
    cmd = sconsPrefix() + testedFile
    executeSuccess(cmd)

  testedClassesStr: str = open(testedFile).read()
  testedClassesList: List[str] = testedClassesStr.strip().split('\n')
  testedClasses = [x.split(' ') for x in testedClassesList]

  # Run the unit tests.

  for pkg, className in testedClasses:

    fullClassName = pkg + '.' + className

    print('Testing ' + str(fullClassName) + '...')

    for target in ('test', 'cover'):

      cmd = sconsPrefix() + target + "-" + className
      status, out, err = executeCmd(cmd)

      if status != 0:

        msg = out + '\n\n\n' + err

        if fullClassName not in errors:
          errors[fullClassName] = {}
        errorDict = errors[fullClassName]
        errorDict[target] = msg

        # If the test failed, don't do the coverage.
        if target == 'test':
          break

#-------------------------------------------------------------------
def runPylint(errors: Dict[str, Dict[str, str]]) -> None:
  '''Run pylint.

  @param errors Dictionary of [className] = errorDict, where errorDict
                is [testName] = errorMsg.
  '''

  # Run Pylint.

  print('Running pylint...')

  cmd = sconsPrefix() + "pylint"
  status, out, err = executeCmd(cmd)

  # Suppress unused variable warning.
  err = err # pylint: disable=self-assigning-variable
  status = status # pylint: disable=self-assigning-variable

  pylintErrors = parsePyLint(out)

  for moduleName, msg in pylintErrors:

    if moduleName not in errors:
      errors[moduleName] = {}
    errorDict = errors[moduleName]
    errorDict['pylint'] = msg

#-------------------------------------------------------------------
def runTslint(errors: Dict[str, Dict[str, str]]) -> None:
  '''Run tylint.

  @param errors Dictionary of [className] = errorDict, where errorDict
                is [testName] = errorMsg.
  '''

  # Run Tslint.

  print('Running tslint...')

  cmd = sconsPrefix() + "tslint"
  ignoredStatus, out, ignoredErr = executeCmd(cmd)

  tslintErrors = parseTsLint(out)

  for moduleName, msg in tslintErrors:

    if moduleName not in errors:
      errors[moduleName] = {}
    errorDict = errors[moduleName]
    errorDict['tslint'] = msg

#-------------------------------------------------------------------
def runBrowserTests(browserTestServer: str) -> Dict[str, str]:
  '''Runs the browser tests.

  @return A dictionary of [test name] = errorMsg.
  '''

  browserErrors: Dict[str, str] = {}

  # Make the list of browser tests to run.

  testSrcFiles = glob('src/browserTest/Test*.py')
  testSrcFiles.sort()
  browserTests = [os.path.basename(x)[:-len('.py')] for x in testSrcFiles]

  # Run browser tests.

  print('Running browser tests...')

  for testName in browserTests:

    print('Running ' + str(testName) + '...')

    cmd = sconsPrefix()
    cmd += ("headless=true browserTestServer=%s browserTest-%s" %
            (browserTestServer, testName))
    status, out, err = executeCmd(cmd)

    if status != 0:

      msg = out + '\n\n\n' + err
      browserErrors[testName] = msg

  return browserErrors

#-------------------------------------------------------------------
def parsePyLint(text: str) -> List[Tuple[str, str]]:
  '''Parses pylint output text, and returns a list of (module, error) tuples.
  '''

  retList = []

  lines = text.strip().split('\n')

  # Find the start of the type-checking section.

  while lines and not lines[0].startswith('Type-checking package '):
    lines.pop(0)
  lines.pop(0)

  # The errors found for each module.

  moduleErrors: Dict[str, List[str]] = OrderedDict()

  # Process the mypy type-checking section.

  while lines and not lines[0].startswith('Linting package '):

    line = lines.pop(0)

    if line.startswith('scons') or line.startswith('Success:'):
      continue

    assert ':' in line, line
    fileName = line[:line.index(':')]
    moduleName = pyFileToModuleName(fileName)

    moduleErrors.setdefault(moduleName, []).append(line)

  # Process the pylint linting section.

  while lines:

    line = lines.pop(0)

    if line.startswith('scons'):
      pass
    elif line.startswith('************* Module'):
      moduleName = line.split(' ')[-1]
      errLines = []

      while (lines and
             not lines[0].startswith('scons') and
             not lines[0].startswith('************* Module')):
        errLines.append(lines.pop(0))

      moduleErrors.setdefault(moduleName, []).extend(errLines)

  for moduleName, errLines in moduleErrors.items():
    retList.append((moduleName, '\n'.join(errLines)))

  return retList

#-------------------------------------------------------------------
def parseTsLint(text: str) -> List[Tuple[str, str]]:
  '''Parses tslint output text, and returns a list of (module, error) tuples.
  '''

  retList = []

  lines = text.strip().split('\n')

  for line in lines:

    if not line.startswith('Error:'):
      continue

    # Remove the prefix.
    line = line[len('Error:'):]

    fileName, ignoredLineNum, ignoredRowNum, ignoredMessage = line.split(':', 3)

    assert fileName.startswith('src/webapp/htmllib/'), fileName
    assert fileName.endswith('.ts'), fileName

    # Remove the prefix and suffix.
    fileName = fileName[len('src/webapp/htmllib/'):]
    fileName = fileName[:-len('.ts')]

    moduleName = '.'.join(fileName.split('/'))

    retList.append((moduleName, line.strip()))

  return retList

#-------------------------------------------------------------------
def writeReport(buildName: str,
                errors: Dict[str, Dict[str, str]],
                browserErrors: Optional[Dict[str, str]]):
  '''Write the error report.

  @param buildName A build name to display.
  @param errors A dictionary of [className] = errorDict, where errorDict
                is [testName] = errorMsg.
  @param browserErrors A dictionary of [testName] = errorMsg, or None if
                       there are no browser tests.
  '''

  outDirName = 'build/report'
  if not os.path.exists(outDirName):
    os.makedirs(outDirName)

  if errors:
    contentHtml = makeTable(errors, outDirName)
  else:
    contentHtml = "No Errors."

  if browserErrors is None:
    browserContentHtml = ""
  elif browserErrors:
    browserContentHtml = '<h3>Browser Test Errors</h3>\n'
    browserContentHtml += makeBrowserTable(browserErrors, outDirName)
  else:
    browserContentHtml = "No Browser Test Errors."

  html = '''<html>
<head>
<title>''' + appLabel() + ''' Build Report</title>
<style>
  body
  {
    text-align: center;
  }
  table
  {
    border-collapse: collapse;
    margin-left: auto;
    margin-right: auto;
  }
  th, td
  {
    border: 1px solid #999;
    padding: 0.5 rem;
  }
</style>
</head>
<body>
<h1>''' + appLabel() + ''' Build Report</h1>
<h4>''' + buildName + '''</h4>
''' + contentHtml + '''
''' + browserContentHtml + '''
</body>
</html>'''

  reportFileName = outDirName + '/report.html'
  open(reportFileName, 'w').write(html)

  print('Wrote file: %s' % reportFileName)

#-------------------------------------------------------------------
def makeBrowserTable(errors: Dict[str, str], outDirName: str) -> str:

  trs = []

  headerHtml = ''
  for header in ('Test',):
    headerHtml += element('th', header)

  trs.append(element('tr', headerHtml))

  for className in sorted(errors):

    trs.append(makeBrowserTableRow(className, errors[className], outDirName))

  table = element('table', '\n'.join(trs))

  return table

#-------------------------------------------------------------------
def makeBrowserTableRow(className: str, errorMsg: str, outDirName: str) -> str:

  errFileName = className + '.txt'
  open(outDirName + '/' + errFileName, 'w').write(errorMsg)

  link = '<a href="' + errFileName + '">' + className + '</a>'

  td = element('td', link)
  tr = element('tr', td)

  return tr

#-------------------------------------------------------------------
def makeTable(errors: Dict[str, Dict[str, str]], outDirName: str) -> str:

  trs = []

  headerHtml = ''
  for header in ('Class', 'Test', 'Cover', 'Py Lint', 'TS Lint'):
    headerHtml += element('th', header)

  trs.append(element('tr', headerHtml))

  for className in sorted(errors):

    trs.append(makeTableRow(className, errors[className], outDirName))

  table = element('table', '\n'.join(trs))

  return table

#-------------------------------------------------------------------
def makeTableRow(className: str,
                 errorDict: Dict[str, str],
                 outDirName: str) -> str:

  html = element('td', className)

  for target in ('test', 'cover', 'pylint', 'tslint'):

    if target in errorDict:

      msg = errorDict[target]

      errFileName = className + '-' + target + '.txt'
      open(outDirName + '/' + errFileName, 'w').write(msg)

      link = '<a href="' + errFileName + '">X</a>'

      html += element('td', link)

    else:

      html += element('td', '')

  html = element('tr', html)

  return html

#-------------------------------------------------------------------
def element(name: str,
            text: str,
            attrs: Optional[Dict[str, str]] = None) -> str:
  '''Return an HTML element <name> containing the specified text.
  '''

  attrText = ''
  if attrs:
    for attrName, attrValue in attrs.items():
      attrText += ' ' + attrName + '="' + attrValue + '"'

  return '<' + name + attrText + '>' + text + '</' + name + '>'

#-------------------------------------------------------------------
def executeSuccess(cmd: str) -> Tuple[str, str]:
  """Execute the specified command.  Halt if it fails.
  """

  status, out, err = executeCmd(cmd)
  if status != 0:
    msg = ('Error running: ' + str(cmd) + '\n' +
           '----------------------------------\n' +
           str(out) + '\n' +
           '----------------------------------\n' +
           str(err) + '\n')
    raise Exception(msg)

  return out, err

#-------------------------------------------------------------------
def executeCmd(cmd: str) -> Tuple[int, str, str]:
  """Execute the specified command.
  """

  # pylint: disable=C0103
  PIPE = subprocess.PIPE
  proc = subprocess.Popen(cmd, shell=True, executable="/bin/bash",
                          stdout=PIPE, stderr=PIPE, encoding='utf-8')
  out, err = proc.communicate()
  status = proc.returncode

  return status, out, err

#-------------------------------------------------------------------
def sconsPrefix() -> str:
  '''Returns the text to run the scons command.
  '''

  appName = BuildSetup.get().appName()
  return ". ./" + appName +  "Env.sh; scons "

#-------------------------------------------------------------------
def appLabel() -> str:
  '''Returns the name of the app, suitable for display to a human.
  '''

  appName = BuildSetup.get().appName()
  return appName[0].upper() + appName[1:]

#-------------------------------------------------------------------
def pyFileToModuleName(fileName: str) -> str:
  '''Converts a python file name to a module name.
  '''

  appName = BuildSetup.get().appName()
  prefix = 'src/python/' + appName + '/'
  suffix = '.py'

  # If the file is not in our src/python tree, we're probably seeing
  # an issue with Axe or some other third-party library, so return
  # generic name 'externalModule'.
  if not fileName.startswith(prefix):
    return 'externalModule'

  assert fileName.startswith(prefix), fileName
  assert fileName.endswith(suffix), fileName

  name = fileName[len(prefix):-len(suffix)]

  name = name.replace('/', '.')

  return name
