#-------------------------------------------------------------------
#  Deploy.py
#
#  The Deploy module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Deploys already-built software to a server.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import os
import sys
from ai.axe.web.build.sconstruct import installations
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
usage = """\
deploy [-q][-v][-n][-storeUser userName][-installUser userName][-noUpdateDb]
       name version tag

Deploys already-built software to the specified server.

The args are:

 * name:     The name of the installation.  This determines where
             the app is deployed to, and the contents of config files.
             Could be 'dev', 'daily', 'demo', or 'test' for example.

 * version:  The version to deploy.

 * tag:      Should be the name of a tag in git, or 'local'.
             If 'local', then the build will be deployed from
             your local 'build' directory rather than from
             the build storage server.

Options are:

 -v           Write verbose output.
 -q           Suppress verbose output.
 -n           Don't execute the commands, just write them to the screen.
 -noUpdateDb The database schema will *not* be upgraded to the current version.
 -storeUser   The user name to use to log in to the tarball store.
 -installUser The user name to use to log in to the install machine.
"""

#-------------------------------------------------------------------
verbose: bool = True

#-------------------------------------------------------------------
def execute() -> None:

  # pylint: disable=W0603
  global verbose
  # pylint: enable=W0603

  showOnly = False

  args = sys.argv[1:]

  if not args or '--help' in args or '-h' in args:
    print(usage)
    sys.exit(0)

  if args and args[0] == "-v":
    args.pop(0)
    verbose = True

  if args and args[0] == "-q":
    args.pop(0)
    verbose = False

  if args and args[0] == "-n":
    args.pop(0)
    showOnly = True

  updateDb: bool = True
  if args and args[0] == '-noUpdateDb':
    args.pop(0)
    updateDb = False

  storeUser = ""
  if args and args[0] == "-storeUser":
    args.pop(0)
    storeUser = args.pop(0)

  installUser = ""
  if args and args[0] == "-installUser":
    args.pop(0)
    installUser = args.pop(0)

  if len(args) != 3:
    print("Error: Wrong number of arguments.")
    sys.exit(1)

  name = args[0]
  version = args[1]
  tag = args[2]

  # Get the commands.

  if tag == 'local':
    isRemote = False
    remoteTag = None
  else:
    isRemote = True
    remoteTag = version + "-" + tag

  cmds = installations.deployCommands(name,
                                      version,
                                      isRemote,
                                      remoteTag,
                                      updateDb,
                                      installUser,
                                      storeUser)

  # Execute the commands.
  for cmd in cmds:
    if showOnly:
      print(cmd)
    else:
      executeCmd(cmd)

#-------------------------------------------------------------------
def executeCmd(cmd: str) -> None:
  """Execute the specified command.
  """

  if verbose:
    print(cmd)

  status = os.system(cmd)
  if status != 0:
    msg = ("Error in script 'deploy'.  The commmand '" + cmd +
           "' had an exit status of " + str(status) + ".")
    raise RuntimeError(msg)
