#-------------------------------------------------------------------
#  TestVersionTreeNode.py
#
#  The TestVersionTreeNode module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the VersionTreeNode class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.web.dbVersion.updateDb import VersionTreeNode
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestVersionTreeNode(AxeSimpleTestCase):
  '''Unit test for the VersionTreeNode class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testTreeString(self) -> None:
    '''Test writing a tree string.
    '''

    root = VersionTreeNode('root', None, '')

    a = VersionTreeNode('a', None, '')

    root.children.append(a)

    a1 = VersionTreeNode('a1', None, '')
    a2 = VersionTreeNode('a2', None, '')
    a3 = VersionTreeNode('a3', None, '')

    a.children.extend([a1, a2, a3])

    a1a = VersionTreeNode('a1a', 'v1', '')
    a1a1 = VersionTreeNode('a1a1', None, '')
    a1a1a = VersionTreeNode('a1a1a', 'v2', '')

    a1.children.append(a1a)
    a1a.children.append(a1a1)
    a1a1.children.append(a1a1a)

    a2a = VersionTreeNode('a2a', None, '')
    a2a1 = VersionTreeNode('a2a1', None, '')

    a2.children.append(a2a)
    a2a.children.append(a2a1)


    expected = '''\
root - 
a - 
* a2 - 
  a2a - 
  a2a1 - 
* a3 - 
a1 - 
a1a (v1) - 
a1a1 - 
a1a1a (v2) - 
'''

    actual = root.toTreeString()

    self.assertEqual(expected, actual, 'simple tree')

  #-----------------------------------------------------------------
  def testFindForks(self) -> None:
    '''Test finding forks in the version history.
    '''

    root = VersionTreeNode('root', None, '')

    self.assertEqual(None, root.findFork(), "None for no children.")

    a = VersionTreeNode('a', None, '')

    root.children.append(a)

    a1 = VersionTreeNode('a1', None, '')
    a2 = VersionTreeNode('a2', None, '')
    a3 = VersionTreeNode('a3', None, '')

    a.children.extend([a1, a2, a3])

    a1a = VersionTreeNode('a1a', 'v1', '')
    a1a1 = VersionTreeNode('a1a1', None, '')
    a1a1a = VersionTreeNode('a1a1a', 'v2', '')
    a1a1b = VersionTreeNode('a1a1a', None, '')

    a1.children.append(a1a)
    a1a.children.append(a1a1)
    a1a1.children.append(a1a1a)
    a1a1.children.append(a1a1b)

    a2a = VersionTreeNode('a2a', None, '')
    a2a1 = VersionTreeNode('a2a1', None, '')

    a2.children.append(a2a)
    a2a.children.append(a2a1)

    # Find the first fork.

    forkNode = root.findFork()

    self.assertEqual(a, forkNode, 'firstFork')

    # Remove the first fork.

    del a.children[1:]


    # Find the second fork.

    forkNode = root.findFork()

    self.assertEqual(a1a1, forkNode, 'secondFork')

  #-----------------------------------------------------------------
  def testFindLeafNode(self) -> None:
    '''Test finding the leaf node of the version history.
    '''

    root = VersionTreeNode('root', None, '')

    a = VersionTreeNode('a', None, '')

    root.children.append(a)

    a1 = VersionTreeNode('a1', None, '')
    a2 = VersionTreeNode('a2', None, '')
    a3 = VersionTreeNode('a3', None, '')

    a.children.extend([a1, a2, a3])

    # Exception for multiple leaf nodes.

    try:

      root.leafNode()
      self.fail("No exception for multiple leaf nodes.")

    except ValueError:

      # Expected result
      pass

    # Remove the excess leaf nodes.

    del a.children[1:]

    # Find the leaf.

    leafNode = root.leafNode()

    self.assertEqual(a1, leafNode, 'leaf')

  #-----------------------------------------------------------------
  def testFindAppVersionLeafNode(self) -> None:
    '''Test finding the appVersion leaf node of the version history.
    '''

    root = VersionTreeNode('root', None, '')

    a = VersionTreeNode('a', None, '')

    root.children.append(a)

    a1 = VersionTreeNode('a1', None, '')
    a2 = VersionTreeNode('a2', None, '')
    a3 = VersionTreeNode('a3', None, '')

    a.children.extend([a1, a2, a3])

    # Exception for multiple leaf nodes.

    try:

      root.appVersionLeafNode()
      self.fail("No exception for multiple leaf nodes.")

    except ValueError:

      # Expected result
      pass


    # Remove the excess leaf nodes.

    del a.children[1:]

    # A version at the leaf.

    a.appVersion = 'v1'
    a1.appVersion = 'v2'

    leafNode = root.appVersionLeafNode()

    self.assertEqual(a1, leafNode, 'leaf')

    # A version in the middle.

    a1.appVersion = ''

    leafNode = root.appVersionLeafNode()

    self.assertEqual(a, leafNode, 'middle')

    # A version at the leaf.

    root.appVersion = 'v0'
    a.appVersion = ''

    leafNode = root.appVersionLeafNode()

    self.assertEqual(root, leafNode, 'root')

  #-----------------------------------------------------------------
  def testFindVersion(self) -> None:
    '''Test finding the node for a specific version.
    '''

    root = VersionTreeNode('root', None, '')

    a = VersionTreeNode('a', 'v1', '')

    root.children.append(a)

    a1 = VersionTreeNode('a1', 'v2', '')

    a.children.append(a1)

    # Find the DB version.

    self.assertEqual(a1, root.findDbVersion('a1'), 'dbVersion')

    # Find the app version.

    self.assertEqual(a, root.findAppVersion('v1'), 'appVersion')

    # Exception for invalid DB version.

    try:

      root.findDbVersion('bad_version')
      self.fail("No exception for bad version.")

    except ValueError:

      pass

    # Exception for invalid app version.

    try:

      root.findAppVersion('bad_version')
      self.fail("No exception for bad version.")

    except ValueError:

      pass

  #-----------------------------------------------------------------
  def testSqlFileNames(self) -> None:
    '''Test finding file names to upgrade to the specified version.
    '''

    root = VersionTreeNode('root', None, None)

    a = VersionTreeNode('a', 'v1', 'file1')

    root.children.append(a)

    a1 = VersionTreeNode('a1', 'v2', 'file2')

    a.children.append(a1)

    # Find the names.

    self.assertEqual(['file1'], root.sqlFileNames(a), 'a')
    self.assertEqual(['file1', 'file2'], root.sqlFileNames(a1), 'a1')

    # No file should be returned for upgrading to the current version.


    self.assertEqual([], root.sqlFileNames(root), 'self root')
    self.assertEqual([], a.sqlFileNames(a), 'self middle')
    self.assertEqual([], a1.sqlFileNames(a1), 'self leaf')

  #-----------------------------------------------------------------
  def testStr(self) -> None:
    '''Test converting to string.
    '''

    # Just make sure no exception is thrown, and get coverage up.
    str(VersionTreeNode('root', None, None))
