#-------------------------------------------------------------------
#  TestVersionDirReader.py
#
#  The TestVersionDirReader module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the VersionDirReader class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.web.dbVersion.updateDb import VersionDirReader
from ai.axe.web.dbVersion.updateDb import VersionTreeNode
from typing import Optional
import tempfile
import shutil
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestVersionDirReader(AxeSimpleTestCase):
  '''Unit test for the VersionDirReader class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testReading(self) -> None:
    '''Test reading a directory.
    '''

    tempDir = tempfile.mkdtemp()

    try:

      fileNames = [
        'updateDb_0_to_1.sql',
        'updateDb_1_to_2_release_v1.sql',
        'updateDb_2_to_3.sql',
        'updateDb_3_to_4_release_v2.sql',
        'updateDb_4_to_5_release_v3.sql',

        'IGNORED_FILE',
        'updateDb_IGNORED_FILE',
        'updateDb_IGNORED_FILE.sql',
        ]

      for fileName in fileNames:
        open(tempDir + '/' + fileName, 'w').write(' ')

      reader = VersionDirReader(tempDir)
      tree = reader.read()

      node = tree

      self.checkNode(node, '0', None, None, 'root')
      node = node.children[0]
      self.checkNode(node, '1', None, 'updateDb_0_to_1.sql', '1')
      node = node.children[0]
      self.checkNode(node, '2', 'v1', 'updateDb_1_to_2_release_v1.sql', '2')
      node = node.children[0]
      self.checkNode(node, '3', None, 'updateDb_2_to_3.sql', '3')
      node = node.children[0]
      self.checkNode(node, '4', 'v2', 'updateDb_3_to_4_release_v2.sql', '4')
      node = node.children[0]
      self.checkNode(node, '5', 'v3', 'updateDb_4_to_5_release_v3.sql', '5')

    finally:

      shutil.rmtree(tempDir)

  #-----------------------------------------------------------------
  def testStr(self) -> None:
    '''Test the str() function.
    '''

    reader = VersionDirReader('/foo/bar')

    # Just check that no exception is thrown.
    str(reader)

  #-----------------------------------------------------------------
  def testNoRoot(self) -> None:
    '''Test errors while reading a directory.
    '''

    tempDir = tempfile.mkdtemp()

    try:

      fileNames = [
        'updateDb_0_to_1.sql',
        'updateDb_1_to_0.sql',
        ]

      for fileName in fileNames:
        open(tempDir + '/' + fileName, 'w').write(' ')

      reader = VersionDirReader(tempDir)

      try:
        reader.read()
        self.fail("No exception for no root.")

      except ValueError:

        # Expected condition.
        pass

    finally:

      shutil.rmtree(tempDir)

  #-----------------------------------------------------------------
  def testMultipleRoots(self) -> None:
    '''Test multiple root errors while reading a directory.
    '''

    tempDir = tempfile.mkdtemp()

    try:

      fileNames = [
        'updateDb_0_to_1.sql',
        'updateDb_a_to_b.sql',
        ]

      for fileName in fileNames:
        open(tempDir + '/' + fileName, 'w').write(' ')

      reader = VersionDirReader(tempDir)

      try:
        reader.read()
        self.fail("No exception for multiple roots.")

      except ValueError:

        # Expected condition.
        pass

    finally:

      shutil.rmtree(tempDir)

  #-----------------------------------------------------------------
  def testMultipleFroms(self) -> None:
    '''Test multiple FROM errors while reading a directory.
    '''

    tempDir = tempfile.mkdtemp()

    try:

      fileNames = [
        'updateDb_0_to_1.sql',
        'updateDb_1_to_2.sql',
        'updateDb_1_to_3.sql',
        ]

      for fileName in fileNames:
        open(tempDir + '/' + fileName, 'w').write(' ')

      reader = VersionDirReader(tempDir)

      try:
        reader.read()
        self.fail("No exception for multiple froms.")

      except ValueError:

        # Expected condition.
        pass

    finally:

      shutil.rmtree(tempDir)

  #-----------------------------------------------------------------
  def checkNode(self,
                node: VersionTreeNode,
                dbVersion: str,
                appVersion: Optional[str],
                sqlFileName: Optional[str],
                msg: str) -> None:
    '''Private helper function to check a node.
    '''

    self.assertEqual(dbVersion, node.dbVersion, msg + ': dbVersion')
    self.assertEqual(appVersion, node.appVersion, msg + ': appVersion')
    self.assertEqual(sqlFileName, node.sqlFileName, msg + ': sqlFileName')
