#-------------------------------------------------------------------
#  VersionTreeNode.py
#
#  The VersionTreeNode class.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the VersionTreeNode class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class VersionTreeNode:
  '''One node of a version history tree.

  This node knows how to update a DB from the parent version to this version.
  It also knows whether this DB schema version was tagged for a release.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               dbVersion: str,
               appVersion: Optional[str],
               sqlFileName: Optional[str]) -> None:
    '''Creates a new VersionTreeNode.
    '''

    # The DB schema version that this node represents.
    self.dbVersion: str = dbVersion

    # The app version that this node represents.
    #
    # This will be None for each incremental change in between software
    # releases.
    #
    self.appVersion: Optional[str] = appVersion

    # The name of the SQL file that will update the parent version to this one.
    #
    # This will be None for the root node (intial version).
    #
    self.sqlFileName: Optional[str] = sqlFileName

    # Child versions.
    #
    # This is a list of VersionTreeNode objects that are all the versions
    # that can be created from this verson.
    self.children: List['VersionTreeNode'] = []

  #----------------------------------------------------------------
  def findDbVersion(self, dbVersion: str) -> 'VersionTreeNode':
    '''Returns the node that matches the specified dbVersion.

    @return The VersionTreeNode with the specified version.
    '''

    if self.dbVersion == dbVersion:
      return self

    elif not self.children:
      msg = "Error: No version spec exists with dbVersion=%s" % dbVersion
      raise ValueError(msg)

    else:
      assert len(self.children) == 1
      return self.children[0].findDbVersion(dbVersion)

  #----------------------------------------------------------------
  def findAppVersion(self, appVersion: str) -> 'VersionTreeNode':
    '''Returns the node that matches the specified appVersion.

    @return The VersionTreeNode with the specified version.
    '''

    if self.appVersion == appVersion:
      return self

    elif not self.children:
      msg = "Error: No version spec exists with appVersion=%s" % appVersion
      raise ValueError(msg)

    else:
      assert len(self.children) == 1
      return self.children[0].findAppVersion(appVersion)

  #----------------------------------------------------------------
  def sqlFileNames(self, targetNode: 'VersionTreeNode') -> List[str]:
    '''Returns all the file names to get from this version to the target node.

    @param targetNode File names to reach this node will be returned.

    @return A list of string file names.  Each is the name of a .sql file
            to be applied to the DB.
    '''

    if self is targetNode:
      return []

    else:
      assert len(self.children) == 1

      child = self.children[0]

      # Only the root is None.
      assert child.sqlFileName is not None

      return [child.sqlFileName] + child.sqlFileNames(targetNode)

  #----------------------------------------------------------------
  def leafNode(self) -> 'VersionTreeNode':
    '''Returns the leaf node.

    If there are multiple leaf nodes, an exception is thrown.

    @return A VersionTreeNode that is the latest version.
    '''

    if not self.children:
      return self

    elif len(self.children) > 1:
      msg = "You called leafNode(), but this tree has multiple leaf nodes."
      raise ValueError(msg)

    else:
      assert len(self.children) == 1
      return self.children[0].leafNode()

  #----------------------------------------------------------------
  def appVersionLeafNode(self) -> Optional['VersionTreeNode']:
    '''Returns the node for the latest version that was included in a release.

    If there are multiple leaf nodes, an exception is thrown.

    @return A VersionTreeNode that is the latest version that has a release
            attached to it, or None if no version has a release.
    '''

    if not self.children:

      return self if self.appVersion else None

    elif len(self.children) > 1:
      msg = "You called leafNode(), but this tree has multiple leaf nodes."
      raise ValueError(msg)

    else:
      assert len(self.children) == 1

      leaf = self.children[0].appVersionLeafNode()

      if leaf:
        # There's a more leaf-ward node with an appVersion so return that.
        return leaf

      else:
        return self if self.appVersion else None

  #----------------------------------------------------------------
  def findFork(self) -> Optional['VersionTreeNode']:
    '''Returns the first node that is a fork in the tree.

    @return The first VersionTreeNode with multiple children.
    '''

    if not self.children:
      return None

    elif len(self.children) > 1:
      return self

    else:
      assert len(self.children) == 1
      return self.children[0].findFork()

  #----------------------------------------------------------------
  def toTreeString(self) -> str:
    '''Returns a string showing this node and its children.

    A tree like:

        root
         |
         a
       /  |  |
      a1  a2 a3
       |  |
      a1a a21
       |
      a1a1 a1a2
       |
      a1a1a

    will produce the following string:

      root
      a
      * a2
        a21
      * a3
      a1
      a1a1
      a1a1a
    '''

    return self.makeTreeString(0, True)

  #----------------------------------------------------------------
  def makeTreeString(self, indentLevel: int, isFirstChild: bool) -> str:
    '''Private implemetation method for toTreeString().

    @param indentLevel How much to indent the string.
    @param isFirstChild If true, this is the first child of its parent.
                        If false, this is a non-first (duplicate) child.
    '''

    indentStr = ' ' * (indentLevel * 2)

    if not isFirstChild:
      indentStr = (' ' * ((indentLevel - 1) * 2)) + '* '

    text = indentStr + self.toTerseString()
    for i, child in enumerate(self.children):

      # The first child will be written last.
      if i == 0:
        continue

      text += child.makeTreeString(indentLevel + 1, isFirstChild=False)

    # Write the first child.
    if self.children:

      child = self.children[0]
      text += child.makeTreeString(indentLevel, isFirstChild=True)

    return text

  #----------------------------------------------------------------
  def toTerseString(self) -> str:
    '''Returns a terse version of this node's data.
    '''

    text = str(self.dbVersion)
    if self.appVersion:
      text += " (%s)" % self.appVersion
    text += " - %s" % self.sqlFileName
    text += '\n'

    return text

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['dbVersion', 'appVersion', 'sqlFileName']

    return StringUtil.formatRepr(self, attrs)
