#-------------------------------------------------------------------
#  VersionDirReader.py
#
#  The VersionDirReader class.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the VersionDirReader class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .VersionTreeNode import VersionTreeNode
from typing import Optional
from typing import Tuple
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class VersionDirReader:
  '''Reads a list of SQL files and creates a VersionTreeNode structure.
  '''

  #-----------------------------------------------------------------
  def __init__(self, dirName: str) -> None:
    '''Creates a new VersionDirReader.
    '''

    # The name of the directory to read.
    self.dirName: str = dirName

  #-----------------------------------------------------------------
  def read(self) -> VersionTreeNode:
    '''Reads the directory and returns a VersionTreeNode structure.

    @return A VersionTreeNode that is the root node for the list
            of versions found in the directory.
    '''

    fileNames = os.listdir(self.dirName)

    nodeTuples = []

    for fileName in fileNames:
      nodeTuple = self.parseFileName(fileName)
      if nodeTuple:
        nodeTuples.append(nodeTuple)

    # Find the root node.

    rootVersions = []

    for fromVersion, node in nodeTuples:
      parents = [x for x in nodeTuples if x[1].dbVersion == fromVersion]
      if not parents:
        rootVersions.append(fromVersion)

    if not rootVersions:
      msg = "Error.  When searching the updateDb_FROM_to_TO.sql files in "
      msg += ("directory %s, " % self.dirName)
      msg += "no file had an initial FROM version (that is, "
      msg += "no file had a FROM version that wasn't also a TO version "
      msg += "of another file).  Perhaps the files all form a cycle?"
      raise ValueError(msg)

    elif len(rootVersions) > 1:
      msg = "Error.  When searching the updateDb_FROM_to_TO.sql files in "
      msg += ("directory %s, " % self.dirName)
      msg += ("%s files " % len(rootVersions))
      msg += "had an initial FROM version (that is, "
      msg += "they had a FROM version that wasn't also a TO version "
      msg += "of another file).  There should only be one initial version. "
      msg += "Initial versions: %s" % ', '.join(rootVersions)
      raise ValueError(msg)

    assert len(rootVersions) == 1

    rootVersion = rootVersions[0]

    nodeTree = VersionTreeNode(rootVersion, None, None)
    parentNode = nodeTree

    foundTuple = True
    while nodeTuples and foundTuple:

      targetVersion = parentNode.dbVersion
      foundTuple = False

      for nodeTuple in nodeTuples[:]:
        fromVersion, node = nodeTuple

        if fromVersion == targetVersion:
          nodeTuples.remove(nodeTuple)
          parentNode.children.append(node)

          if foundTuple:
            msg = "Error.  When searching the updateDb_FROM_to_TO.sql files in "
            msg += ("directory %s, found " % self.dirName)
            msg += ("multiple files with FROM version '%s'." % targetVersion)
            raise ValueError(msg)

          foundTuple = True

      if parentNode.children:
        parentNode = parentNode.children[0]

    assert not nodeTuples, nodeTuples

    return nodeTree

  #----------------------------------------------------------------
  def parseFileName(self,
                    fileName: str) -> Optional[Tuple[str, VersionTreeNode]]:
    '''Parses the specified file name into a VersionTreeNode object.

    @fileName A file name string.

    @return A (fromVersion, VersionTreeNode object) tuple,
            or None if the string cannot be parsed as an updateDb file.
    '''

    fullFileName = fileName

    if not fileName.startswith('updateDb_'):
      return None
    else:
      fileName = fileName[len('updateDb_'):]

    if not fileName.endswith('.sql'):
      return None
    else:
      fileName = fileName[:-len('.sql')]

    if '_to_' not in fileName:
      return None

    fromVersion, toVersion = fileName.split('_to_', 1)
    appVersion = None

    if '_release_' in toVersion:
      toVersion, appVersion = toVersion.split('_release_', 1)

    return fromVersion, VersionTreeNode(toVersion, appVersion, fullFileName)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['dirName']

    return StringUtil.formatRepr(self, attrs)
