#-------------------------------------------------------------------
#  AppVersionDb.py
#
#  The AppVersionDb module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Database access functions for the AppVersion class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.web.app import AppSetup
from . import AppVersionClass
from sqlalchemy.orm.session import Session
from sqlalchemy import Table
from typing import Any
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def createTable(session: Session) -> None:
  '''Creates an AppVersion table.

  @param session The DB to write to.
  '''

  sql = AppSetup.get().databaseMgr().generateDdl(tables=[getTable()])

  session.execute(sql)

#-------------------------------------------------------------------
def tableExists(session: Session) -> bool:
  '''Returns true if the AppVersion table exists.

  @param session The DB to write to.
  '''

  sql = '''\
select count(*)
from information_schema.tables 
where table_name = :tableName and
      (table_schema = :tableSchemaPostgres or
       table_schema = :tableSchemaMsSql)
'''

  params = {
    "tableName": getTable().name,
    "tableSchemaMsSql": "dbo",
    "tableSchemaPostgres": "public",
    }

  result = session.execute(sql, params)

  numTables = result.fetchall()[0][0]
  exists = numTables > 0

  return exists

#-------------------------------------------------------------------
def update(session: Session, appVersion: str, dbVersion: str) -> None:
  '''Updates the versions stored in the database.

  @param session The database to write to.
  @param appVersion The appVersion string to write.
  @param dbVersion The dbVersion string to write.
  '''

  sql = "update app_versions set db_version=:dbVersion, app_version=:appVersion"
  params = {
    "appVersion": appVersion,
    "dbVersion": dbVersion,
    }

  result = session.execute(sql, params)

  print("app version update result:", result)

#-------------------------------------------------------------------
def count(session: Session) -> int:
  '''Returns the number of AppVersion objects in the database.

  @param session The database to read from.

  @return The number of AppVersion objects.
  '''

  return session.query(AppVersionClass.get()).count()

#-------------------------------------------------------------------
def read(session: Session) -> Any:
  '''Reads the AppVersion from the database.

  @param session The database to read from.

  @return The AppVersion object.
  '''

  appVersions = session.query(AppVersionClass.get()).all()

  assert len(appVersions) == 1, len(appVersions)

  return appVersions[0]

#-------------------------------------------------------------------
def write(session: Session, appVersion: Any) -> None:
  '''Writes an AppVersion to the database.

  @param session The database to write to.
  @param appVersion The AppVersion object to write.
  '''

  session.add(appVersion)

#-------------------------------------------------------------------
def getTable() -> Table:
  '''Returns the AppVersion __table__.
  '''

  # pylint: disable=E1101
  return AppVersionClass.get().__table__
