#-------------------------------------------------------------------
#  Log.py
#
#  The Log class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Log class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import logging
import os
import sys
from ..config.LogConfig import LogConfig
from ai.axe.logging import DayFileHandler
from ai.axe.logging import UtcFormatter
from logging import Logger
from logging import StreamHandler
from logging import Handler
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class Log:
  '''The Axe app log.
  '''

  # The name to use for the log file.
  logFileBaseName: str = "axe"

  # The logger object to write info/warn/error messages to.
  logger: Optional[Logger] = None

  # The logger object to log web actions to.
  actionLogger: Optional[Logger] = None

  # The logger object to write debug SQL messages to.
  dbLogger: Optional[Logger] = None

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new Log.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def debug(msg: str, *args, **kwargs) -> None:
    '''Writes a debug message.

    @param msg The message to write.
    '''

    assert Log.logger is not None

    Log.logger.debug(msg, *args, **kwargs)

  #-----------------------------------------------------------------
  @staticmethod
  def info(msg: str, *args, **kwargs) -> None:
    '''Writes an info message.

    @param msg The message to write.
    '''

    assert Log.logger is not None

    Log.logger.info(msg, *args, **kwargs)

  #-----------------------------------------------------------------
  @staticmethod
  def warn(msg: str, *args, **kwargs) -> None:
    '''Writes a warning message.

    @param msg The message to write.
    '''

    assert Log.logger is not None

    Log.logger.warning(msg, *args, **kwargs)

  #-----------------------------------------------------------------
  @staticmethod
  def error(msg: str, *args, **kwargs) -> None:
    '''Writes an error message.

    @param msg The message to write.
    '''

    assert Log.logger is not None

    Log.logger.error(msg, *args, **kwargs)

  #-----------------------------------------------------------------
  @staticmethod
  def logException(msg: str, exceptionStr: str) -> None:
    '''Logs an exception.

    @param msg The message to write to the log.
    @param exceptionStr The string returned by traceback.format_exc().
    '''

    msg += "\n"
    msg += "-------------------------------------------------\n"
    msg += str(exceptionStr)

    Log.error(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def logAction(actionName: str,
                msg: str,
                wasError: bool,
                userName: str,
                userHost: str) -> None:
    '''Logs an action.

    @param actionName The action performed.
    @param msg The message to write to the log.
    @param wasError Whether there was an error during the action.
    @param userName The name of the user running the action.
    @param userHost The host the user connected from.
    '''

    assert Log.actionLogger is not None
    assert Log.logger is not None

    if wasError:
      status = "had an error during"
    else:
      status = "succeeded at"

    msg = ("User " + userName + " " + userHost + " " + status +
           " action: " + actionName + '\n' + msg)

    # All actions go to the action log.
    Log.actionLogger.info(msg)

    # Error actions also go the error log.
    if wasError:
      Log.logger.error(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def initWebLog(logConfig: LogConfig, errorsToStdout: bool) -> None:
    '''Initializes the logging system.

    @param logConfig a LogConfig object.
    @param errorsToStdout Whether to log error to stdout.
    '''

    Log.initLog(logConfig.webDir,
                logConfig.debug,
                logConfig.sql,
                errorsToStdout)


  #-----------------------------------------------------------------
  @staticmethod
  def initCommandLog(logConfig: LogConfig) -> None:
    '''Initializes the logging system.

    @param logConfig a LogConfig object.
    '''

    Log.initLog(logConfig.commandDir,
                logConfig.debug,
                logConfig.sql,
                False)


  #-----------------------------------------------------------------
  @staticmethod
  def initUnitTestLog(debugOn: bool, sqlOn: bool) -> None:
    '''Initializes the logging system.

    @param logConfig a LogConfig object.
    '''

    Log.initLog(None,
                debugOn,
                sqlOn,
                True)


  #-----------------------------------------------------------------
  @staticmethod
  def initLog(logDir: Optional[str],
              debugOn: bool,
              sqlOn: bool,
              errorsToStdout: bool) -> None:
    '''Initializes the logging system.
    '''

    # If the log directory doesn't exist, create it now.
    if logDir and not os.path.exists(logDir):
      os.makedirs(logDir)

    level = logging.INFO
    if debugOn:
      level = logging.DEBUG
    Log.logger = Log.createLogger(logDir, 'error', level, errorsToStdout)

    Log.actionLogger = Log.createLogger(logDir, 'action', logging.INFO)

    if sqlOn:
      Log.dbLogger = Log.createLogger(logDir, 'sqlalchemy.engine', logging.INFO)

  #-----------------------------------------------------------------
  @staticmethod
  def setLogFileBaseName(logFileBaseName: str) -> None:
    '''Sets the name of the log files.
    '''

    Log.logFileBaseName = logFileBaseName

  #-----------------------------------------------------------------
  @staticmethod
  def createLogger(logDir: Optional[str],
                   name: str,
                   level: int,
                   writeToStdout: bool = False) -> Logger:

    logger = logging.getLogger(name)
    logger.setLevel(level)

    fmt = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    formatter = UtcFormatter(fmt)

    if logDir:
      fileName = os.path.join(logDir, Log.logFileBaseName + '-' + name + '.log')
      handler: Handler = DayFileHandler(fileName)

      handler.setFormatter(formatter)
      logger.addHandler(handler)

    if writeToStdout:
      handler = StreamHandler(sys.stderr)
      handler.setFormatter(formatter)
      logger.addHandler(handler)

    return logger
