#-------------------------------------------------------------------
#  TestConfig.py
#
#  The TestConfig module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the Config class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest import AxeSimpleTestCase
from ai.axe.web.core import AxeException
from ai.axe.web.config import Config
from ai.axe.web.config import ConfigSection
from ai.axe.web.config.ConfigTypes import ConfigValueType
from typing import cast
from typing import Iterable
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class PersonConfig(ConfigSection):
  '''A class for testing:  information about a person.
  '''

  def __init__(self, firstName, lastName, middleName=None) -> None:

    ConfigSection.__init__(self)
    self.firstName = firstName
    self.lastName = lastName
    self.middleName = middleName

  @staticmethod
  def configKeys() -> Iterable[Tuple[str, ConfigValueType, bool]]:

    return (("firstName", str, False),
            ("lastName", str, False),
            ("middleName", str, True))

#===================================================================
class TestConfig(AxeSimpleTestCase):
  '''Unit test for the Config class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.

    Config.removeAllSections()
    Config.addSection("person", PersonConfig)

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.

    Config.removeAllSections()

  #-----------------------------------------------------------------
  def testTypes(self) -> None:
    '''Test conversion to types.
    '''

    # Bool

    data = {"itemTrue": "yes", "itemFalse": "no"}

    value = Config.getType(data, "itemTrue", bool)
    self.assertEqual(True, value, "bool")
    value = Config.getType(data, "itemFalse", bool)
    self.assertEqual(False, value, "bool")


    data = {"item": "blah"}

    try:
      Config.getType(data, "item", bool)
      self.fail("No exception for mal-formed bool.")
    except AxeException:
      pass

    # Int

    data = {"theAnswer": "42"}

    value = Config.getType(data, "theAnswer", int)
    self.assertEqual(42, value, "int")

    data = {"item": "blah"}

    try:
      Config.getType(data, "item", int)
      self.fail("No exception for mal-formed int.")
    except AxeException:
      pass

    # String

    data = {"someString": "foo"}

    value = Config.getType(data, "someString", str)
    self.assertEqual("foo", value, "str")

    # String List

    data = {"someStrings": "foo1,foo2, foo3 "}

    value = Config.getType(data, "someStrings", list)
    self.assertEqual(["foo1", "foo2", "foo3"], value, "list")

    try:
      Config.getType(data, "someStrings", float)          # type: ignore
      self.fail("No exception for an unsupported type.")
    except AxeException:
      pass

  #-----------------------------------------------------------------
  def testValidation(self) -> None:
    '''Test validation of extra/missing keys.
    '''

    # Valid keys, without the optional.

    text = "person.firstName=Joe\nperson.lastName=Smith\n"
    Config.readProperties(text, "testConfig")

    # Valid keys, with the optional.

    text = "person.firstName=Joe\nperson.lastName=Smith\nperson.middleName=X\n"
    Config.readProperties(text, "testConfig")

    # Extra keys.

    text = "person.firstName=Joe\nperson.lastName=Smith\na=a\nb=b\n"

    try:
      Config.readProperties(text, "testConfig")
      self.fail("No exception for extra keys.")
    except AxeException:
      pass

    # Missing keys.

    text = "\n"

    try:
      Config.readProperties(text, "testConfig")
      self.fail("No exception for missing keys.")
    except AxeException:
      pass

    # Extra and missing keys.

    text = "a=a\nb=b\n"

    try:
      Config.readProperties(text, "testConfig")
      self.fail("No exception for extra and missing keys.")
    except AxeException:
      pass

    # Invalid file format.

    text = "a b=\n"

    try:
      Config.readProperties(text, "testConfig")
      self.fail("No exception for malformed file.")
    except AxeException:
      pass

  #-----------------------------------------------------------------
  def testGetConfig(self) -> None:
    '''Test getting a config object.
    '''

    # Get by name.

    text = "person.firstName=Joe\nperson.lastName=Smith\nperson.middleName=X\n"

    config = Config(Config.readProperties(text, "testConfig"))

    personConfig: PersonConfig = cast(PersonConfig,
                                      config.getConfigByName("person"))

    self.assertEqual("Joe", personConfig.firstName, "firstName")
    self.assertEqual("Smith", personConfig.lastName, "lastName")
    self.assertEqual("X", personConfig.middleName, "middleName")

    # Optional key present.

    text = "person.firstName=Joe\nperson.lastName=Smith\nperson.middleName=X\n"

    config = Config(Config.readProperties(text, "testConfig"))

    personConfig = config.getConfig(PersonConfig)

    self.assertEqual("Joe", personConfig.firstName, "firstName")
    self.assertEqual("Smith", personConfig.lastName, "lastName")
    self.assertEqual("X", personConfig.middleName, "middleName")


    # Optional key not present.

    text = "person.firstName=Joe\nperson.lastName=Smith\n\n"

    config = Config(Config.readProperties(text, "testConfig"))

    personConfig = config.getConfig(PersonConfig)

    self.assertEqual("Joe", personConfig.firstName, "firstName")
    self.assertEqual("Smith", personConfig.lastName, "lastName")
    self.assertEqual(None, personConfig.middleName, "middleName")


    # Invalid config type.

    try:
      config.getConfigByName("invalid")
      self.fail("No exception for invalid section name.")
    except AxeException:
      pass

    class UnregisteredConfig(ConfigSection):
      @staticmethod
      def configKeys() -> Iterable[Tuple[str, ConfigValueType, bool]]:
        return ()

    try:
      config.getConfig(UnregisteredConfig)
      self.fail("No exception for invalid class.")
    except AxeException:
      pass
