#-------------------------------------------------------------------
#  DbConfig.py
#
#  The DbConfig class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the DbConfig class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import ai.axe.db.DbConfig
from ai.axe.db import ConnectionUrl
from .ConfigSection import ConfigSection
from .ConfigTypes import ConfigValueType
from typing import Iterable
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class DbConfig(ai.axe.db.DbConfig, ConfigSection):
  '''Database configuration.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               url: str,
               remoteUnitTests: bool,
               poolSize: int,
               poolOverflow: int,
               poolRecycle: int,
               poolTimeout: int) -> None:
    '''Creates a new DbConfig.
    '''

    ai.axe.db.DbConfig.__init__(self,
                                url,
                                remoteUnitTests,
                                poolSize,
                                poolOverflow,
                                poolRecycle,
                                poolTimeout)
    ConfigSection.__init__(self)

  #-----------------------------------------------------------------
  @staticmethod
  def configKeys() -> Iterable[Tuple[str, ConfigValueType, bool]]:
    '''Returns a list of those keys that must be present in the config file.

    If a required (isOptional=False) key is missing from the config file,
    the app will throw an error when the config file is read.

    @return A list of (keyName, type, isOptional) tuples, where:
            'keyName' is the string name of the key as it must appear
                      in the file;
            'type' is one of: int, str, bool, list;
            'isOptional' is a boolean: if False the key must be in the file,
                         if True the key may be absent.
    '''

    keys = (
      ('url', str, False),
      ('remoteUnitTests', bool, False),
      ('poolSize', int, False),
      ('poolOverflow', int, False),
      ('poolRecycle', int, False),
      ('poolTimeout', int, False),
      )

    return keys

  #-----------------------------------------------------------------
  def unitTestConfig(self) -> 'DbConfig':
    '''Returns the DB configuration for unit testing, based on this config.

    @return a new DbConfig object.
    '''

    url: str = ConnectionUrl.unittestDbName(self.url, self.remoteUnitTests)

    # Build the new config.

    unitDbConfig = DbConfig(url,
                            self.remoteUnitTests,
                            self.poolSize,
                            self.poolOverflow,
                            self.poolRecycle,
                            self.poolTimeout)

    return unitDbConfig
