#-------------------------------------------------------------------
#  ConfigParser.py
#
#  The ConfigParser class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ConfigParser class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.web.core.AxeException import AxeException
from collections import OrderedDict
from typing import Dict
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ConfigParser:
  '''Parser for configuration files.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new ConfigParser.
    '''

    self.text: str = ""
    self.data: Dict[str, str] = OrderedDict()
    self.index: int = -1
    self.column: int = -1
    self.line: int = -1

  #-----------------------------------------------------------------
  def parse(self, text: str) -> Dict[str, str]:
    '''Parses the specified config file contents.

    @param text The config file text to  parse.

    @return A dict of keys and values.
    '''

    self.text = text

    # The config data dictionary.
    self.data = OrderedDict()

    # The index of the last character returned by nextChar().
    self.index = -1

    self.column = 0
    self.line = 1

    self.skipWhite()

    while self.peekChar():
      self.readKeyValue()
      self.skipWhite()

    return self.data

  #-----------------------------------------------------------------
  def skipWhite(self) -> None:
    '''Skips to the next non-comment char, or the end of the stream.
    '''

    while True:

      ch = self.peekChar()
      if not ch:
        return
      elif not ch.isspace():
        return
      else:
        self.nextChar()

  #-----------------------------------------------------------------
  # pylint: disable=R0912
  def readKeyValue(self) -> None:
    '''Reads a key-value pair.
    '''

    key = ''
    foundEquals = False

    # Read the key.
    while True:
      ch = self.nextChar(allowComments=False)
      if not ch:
        msg = self.errorPrefix() + 'End of file mid-key.'
        raise AxeException(msg)
      elif ch.isspace():
        break
      elif ch == '=':
        foundEquals = True
        break
      else:
        key += ch

    # Find the equals sign, if it wasn't right next to the key.
    while not foundEquals:
      ch = self.nextChar(allowComments=False)
      if not ch:
        msg = (self.errorPrefix() + 'Saw a key, but reached the end ' +
               'of the file before finding an equals sign.')
        raise AxeException(msg)
      elif ch.isspace():
        continue
      elif ch == '=':
        foundEquals = True
      else:
        msg = (self.errorPrefix() + 'After key, found ' +
               'extra text before finding an equals sign.')
        raise AxeException(msg)

    # Read the value.

    value = ''
    while True:

      ch = self.nextChar(commentsAreJustText=True)
      if not ch:
        break
      elif ch == '\n':
        break
      elif ch == '\\' and self.peekChar(skipComments=False) == '\n':
        ch = self.nextChar(commentsAreJustText=True)

      assert isinstance(ch, str), ch
      value += ch

    value = value.strip()

    # Eat whitespace.

    self.skipWhite()

    self.data[key] = value

  #-----------------------------------------------------------------
  def errorPrefix(self) -> str:
    '''Returns the start of an error message.
    '''
    msg = ('Error at line ' + str(self.line) + ', ' +
           'column ' + str(self.column) + ': ')
    return msg

  #-----------------------------------------------------------------
  def peekChar(self, skipComments: bool = True) -> Optional[str]:
    '''Returns the next character, but doesn't consume it.

    @return The next character, or None if there are no more characters.
    '''

    peekIndex = self.index + 1

    if peekIndex >= len(self.text):
      return None

    ch = self.text[peekIndex]

    if not skipComments:
      return ch

    inComment = False
    if ch == '#':
      inComment = True

    # Find the end of the comment.
    while inComment:
      peekIndex += 1
      if peekIndex >= len(self.text):
        return None

      ch = self.text[peekIndex]
      if ch == '\n':
        inComment = False

    if peekIndex >= len(self.text):
      return None
    return self.text[peekIndex]

  #-----------------------------------------------------------------
  def nextChar(self,
               commentsAreJustText: bool = False,
               allowComments: bool = True) -> Optional[str]:
    '''Returns the next non-comment character.

    @param commentsAreJustTest If False, comments are special and will
                               be skipped.
                               If True, they will be returned as normal text.
    @param allowComments If False, an exception will be thrown if a comment
                         is found.

    @return The next character, or None if there are no more characters.
    '''

    inComment = False

    while True:
      self.index += 1

      if self.index >= len(self.text):
        return None

      ch = self.text[self.index]

      self.column += 1
      if ch == '\n':
        self.line += 1
        self.column = 0

      if inComment:
        if ch == '\n':
          return ch
      else:
        if ch == '#' and not commentsAreJustText:
          inComment = True
          if not allowComments:
            msg = self.errorPrefix() + "Comment not allowed here."
            raise AxeException(msg)
        else:
          return ch
