#-------------------------------------------------------------------
#  Config.py
#
#  The Config class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Config class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .ConfigParser import ConfigParser
from .ConfigSection import ConfigSection
from .ConfigTypes import ConfigValue
from .ConfigTypes import ConfigValueType
from .DbConfig import DbConfig
from .LogConfig import LogConfig
from ai.axe.web.core import AxeException
from ai.axe.util import FileUtil
from ai.axe.util import StringUtil
from collections import OrderedDict
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
from typing import Type
from typing import TypeVar
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
# Generic type used by the static constructor function.
ConcreteConfig = TypeVar('ConcreteConfig', bound=ConfigSection)

#===================================================================
class Config:
  '''RAM configuration information.
  '''

  #-----------------------------------------------------------------
  # The environment variable which contains the file name of the config file.
  configFileEnvVar: str = "NOT_SET_YET_CONFIG"

  #-----------------------------------------------------------------
  # Sections which every config file must have.
  configSectionClasses: Dict[str, Type[ConfigSection]] = OrderedDict()

  #-----------------------------------------------------------------
  def __init__(self, configDict: Dict[str, ConfigValue]) -> None:
    '''Creates a new Config.

    @param configDict A string:string dictionary of config keys and values.
    '''

    self.data: Dict[str, ConfigValue] = configDict

  #-----------------------------------------------------------------
  @staticmethod
  def initDefaults() -> None:
    '''Creates default configuration file sections.

    These can be replaced by user-defined subclasses, if desired.
    '''

    Config.addSection("db", DbConfig)
    Config.addSection("log", LogConfig)

  #-----------------------------------------------------------------
  @staticmethod
  def addSection(name: str, configSectionClass: Type[ConfigSection]) -> None:
    '''Adds a section to the config.

    @param name A String name of the section that will be used with
                the getConfig() method to return objects of the
                configSectionClass type.
    @param configSectionClass A ConfigSection subclass that will be part
                              of your app's config.
    '''

    Config.configSectionClasses[name] = configSectionClass

  #-----------------------------------------------------------------
  @staticmethod
  def removeAllSections() -> None:
    '''Removes all sections that have been added via the addSection() method.
    '''

    Config.configSectionClasses.clear()

  #-----------------------------------------------------------------
  @staticmethod
  def readConfig() -> 'Config':
    '''Returns the config pointed to by the config environment variable.

    @return A Config object containing the contents of the config file.
    '''

    envVarName: str = Config.configFileEnvVar

    if envVarName not in os.environ:
      msg = 'Error: %s environment variable not defined.' % envVarName
      raise AxeException(msg)

    configFileName: str = os.environ[envVarName]
    configText: str = FileUtil.readAll(configFileName)

    configDict = Config.readProperties(configText, configFileName)

    return Config(configDict)

  #-----------------------------------------------------------------
  def getConfigByName(self, name: str) -> ConfigSection:
    '''Returns the specified config.

    @param name The config to be returned, as specified with the addSection()
                'name' parameter.

    @return A ConfigSection object of the type previously registered with the
            addSection() method.
    '''

    sectionName = name

    if sectionName not in Config.configSectionClasses:
      msg = "Invalid config file section name '%s'.  " % sectionName
      msg += "Valid sections are: "
      msg += ", ".join(Config.configSectionClasses.keys())
      raise AxeException(msg)

    clazz = Config.configSectionClasses[sectionName]

    return self.getConfig(clazz)

  #-----------------------------------------------------------------
  def getConfig(self, clazz: Type[ConcreteConfig]) -> ConcreteConfig:
    '''Returns the specified config.

    @param clazz The config to be returned, as specified with the addSection()
                 class parameter.  This will be class such as
                 LogConfig or DbConfig.

    @return A ConfigSection object of the type previously registered with the
            addSection() method.
    '''

    if clazz not in Config.configSectionClasses.values():
      msg = "Invalid config file section class '%s'.  " % clazz
      msg += "Valid sections are: "
      msg += ", ".join((str(x) for x in Config.configSectionClasses.values()))
      raise AxeException(msg)

    # Find the name of the section.

    srcItems = Config.configSectionClasses.items()
    nameMap: Dict[Type[ConfigSection], str] = {v: k for k, v in srcItems}
    sectionName: str = nameMap[clazz]

    # Build up the constructor values.

    keys = clazz.configKeys()

    args: List[Optional[ConfigValue]] = []

    for keyName, ignoredKeyType, isOptional in keys:

      keyName = sectionName + "." + keyName

      if isOptional and keyName not in self.data:
        args.append(None)
      else:
        args.append(self.data[keyName])

    # Create the object.

    configSection = clazz(*args) # type: ignore

    return configSection

  #-----------------------------------------------------------------
  @staticmethod
  def getType(dataStrs: Dict[str, str],
              key: str,
              keyType: ConfigValueType) -> ConfigValue:
    '''Reads a single property as the specified type.

    An exception will be thrown if the property can't be converted to that type.
    '''

    if keyType == int:
      return Config.getInt(dataStrs, key)
    elif keyType == bool:
      return Config.getYesNo(dataStrs, key)
    elif keyType == list:
      return Config.getStringList(dataStrs, key)
    elif keyType == str:
      return dataStrs[key]
    else:
      msg = "Key %s has invalid type '%s'.  " % (key, keyType)
      msg += "Must be one of: bool, int, str, list."
      raise AxeException(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def getInt(dataStrs: Dict[str, str], key: str) -> int:
    '''Reads a single property as an int.

    An exception will be thrown if the property can't be converted to int.
    '''

    valueStr = dataStrs[key]

    try:
      value = int(valueStr)
    except ValueError:
      msg = ("Error: The '" + key + "' value is not a valid " +
             "number.  Value: '" + valueStr + "'.")
      raise AxeException(msg)

    return value

  #-----------------------------------------------------------------
  @staticmethod
  def getYesNo(dataStrs: Dict[str, str], key: str) -> bool:
    '''Reads a single yes/no property as a boolean.

    An exception will be thrown if the property can't be converted to bool.
    '''

    valueStr = dataStrs[key]

    if valueStr == "yes":
      return True
    elif valueStr == "no":
      return False
    else:
      msg = ("Error: The '" + key + "' value must be either 'yes' or " +
             "'no'.  Value: '" + valueStr + "'.")
      raise AxeException(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def getStringList(dataStrs: Dict[str, str], key: str) -> List[str]:
    '''Reads a single property as a list of strings.
    '''

    valueStr = dataStrs[key]

    ret = []

    if valueStr:
      for value in valueStr.split(","):
        ret.append(value.strip())

    return ret

  #-----------------------------------------------------------------
  @staticmethod
  def readProperties(text: str, configFileName: str) -> Dict[str, ConfigValue]:
    '''Returns the Config properties contained in the specified stream.

    @param text A string containing configuration information.
    @param configFileName The file name.  Used for error messages.

    @return A dictionary of string keys/values found in the stream.
    '''

    # Parse the file text.

    parser = ConfigParser()
    try:
      dataStrings = parser.parse(text)
    except Exception as e:
      msg = "Error loading config file '" + configFileName + "'."
      raise AxeException(msg, e)

    # Make lists of mandatory and optional keys.

    configKeys: List[Tuple[str, ConfigValueType, bool]] = []
    for sectionName, configSectionClass in Config.configSectionClasses.items():
      for keyName, keyType, isOptional in configSectionClass.configKeys():
        keyName = sectionName + "." + keyName
        configKeys.append((keyName, keyType, isOptional))

    mandatoryKeys: List[str] = []
    optionalKeys: List[str] = []

    for keyName, keyType, isOptional in configKeys:
      if isOptional:
        optionalKeys.append(keyName)
      else:
        mandatoryKeys.append(keyName)

    # Check for any missing or extra keys and create a nice error message.

    fileKeys: List[str] = list(dataStrings.keys())

    extraKeys = fileKeys[:]
    for key in mandatoryKeys:
      if key in extraKeys:
        extraKeys.remove(key)
    for key in optionalKeys:
      if key in extraKeys:
        extraKeys.remove(key)

    missingKeys = mandatoryKeys[:]
    for key in fileKeys:
      if key in missingKeys:
        missingKeys.remove(key)

    if extraKeys or missingKeys:
      msg = "Configuration error in file '" + configFileName + "'.\n"

      if extraKeys:
        msg += "  Invalid keys:\n"
        for k in extraKeys:
          msg += "    '" + k + "'\n"

      if missingKeys:
        msg += "  Missing keys:\n"
        for k in missingKeys:
          msg += "    '" + k + "'\n"

      raise AxeException(msg)

    # Convert the strings to objects.

    data: Dict[str, ConfigValue] = OrderedDict()

    for keyName, keyType, isOptional in configKeys:

      if keyName in dataStrings:
        keyValue = Config.getType(dataStrings, keyName, keyType)
        data[keyName] = keyValue

    return data

  #----------------------------------------------------------------
  @staticmethod
  def setEnvVarPrefix(envVarPrefix: str) -> None:
    '''Sets the name of the envirnment variable with the file name.

    This prefix will have '_CONFIG' appended to it, and that
    environment variable will be looked up at run time to fnd the
    path to the config file to read.
    '''

    Config.configFileEnvVar = envVarPrefix + "_CONFIG"

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['data']

    return StringUtil.formatRepr(self, attrs)

#----------------------------------------------------------------
# Set up the default config file sections.
Config.initDefaults()
