#-------------------------------------------------------------------
#  UpdateDb.py
#
#  The UpdateDb module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The UpdateDb command.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .AxeCommand import AxeCommand
from .AxeCommandBase import AxeCommandBase
from ai.axe.web.dbVersion.updateDb import DbUpdater
from sqlalchemy.orm.session import Session
from typing import List
from typing import Optional
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
@AxeCommand()
class UpdateDb(AxeCommandBase):
  '''The UpdateDb command object.
  '''

  #-----------------------------------------------------------------
  name = 'updateDb'

  #-----------------------------------------------------------------
  description = 'Updates the database to the current version.'

  #-----------------------------------------------------------------
  usage = '''updateDb [-release] [-setVersion ver] [-maxVersion ver] [-q] [-run]

Updates the database to the current version.

By default, the database is updated to the latest DB schema version
for which there is a updateDb file in the SQL files directory.
If the '-release' flag is provided, the database is instead upgraded
to the DB schema that matches the current app version.

To avoid accidentally writing to the database, no updating will be done
without the '-run' flag.

The -setVersion VERSION option tells 'updateDb' to record that your
database is currently at the specified version.  This is useful
if you have manually set up your database to be at a certain version,
and now want updateDb to update it to the current version.

The -maxVersion VERSION option tells 'updateDb' to update the database
only to the specified DB schema version and not past it.

Normally, the DDL SQL is written to stdout.  The '-q' flag prevents this.
'''

  #---------------------------------------------------------------
  def run(self,
          session: Session,
          args: List[str],
          wantsTabCompletion: bool) -> None:

    if wantsTabCompletion:
      print("-run -q -release -setVersion -maxVersion")
      return

    hasRun: bool = False
    isVerbose: bool = True
    useRelease: bool = False
    dbVersionToSet: Optional[str] = None
    maxDbVersion: Optional[str] = None

    while args:
      arg = args.pop(0)

      if arg == '-run':
        hasRun = True
      elif arg == '-q':
        isVerbose = False
      elif arg == '-release':
        useRelease = True
      elif arg == '-setVersion':
        if not args:
          print("Error: -setVersion must be followed by a version string.")
          sys.exit(1)
        dbVersionToSet = args.pop(0)
      elif arg == '-maxVersion':
        if not args:
          print("Error: -maxVersion must be followed by a version string.")
          sys.exit(1)
        maxDbVersion = args.pop(0)
      else:
        print("Unknown argument: ", arg)
        print()
        sys.exit(1)

    updater = DbUpdater(session, isVerbose)
    updater.update(useRelease, hasRun, dbVersionToSet, maxDbVersion)
