#-------------------------------------------------------------------
#  BatchCommand.py
#
#  The BatchCommand class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the BatchCommand class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .AxeCommandBase import AxeCommandBase
from ai.axe.web.core import AxeException
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
from typing import Type
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class BatchCommand:
  '''Decorator that marks a class as an Axe batch command.

  Every @AxeCommand-decorated class must have the method:

    run()

  In addition, a @BatchCommand-decorated class must have additional methods:

    init()
    fini()

  The init() method must return a list of string labels, one for each time
  run() is to be called.  These labels will be used for logging.

  The fini() method is called last, and has no return value.
  '''

  #-----------------------------------------------------------------
  # Static list of all registered commands.
  registeredCommands: List[Type[AxeCommandBase]] = []

  #-----------------------------------------------------------------
  @staticmethod
  def find(commandName: str) -> Optional[Type[AxeCommandBase]]:
    '''Returns the command with the specified name, or None.
    '''

    for cmd in BatchCommand.registeredCommands:
      if cmd.name == commandName:
        return cmd

    return None

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new BatchCommand.
    '''

    pass

  #-----------------------------------------------------------------
  def decorate(self, clazz: Type[AxeCommandBase]) -> Type[AxeCommandBase]:
    '''Sets up a class to be executed as a batch command.

    @param clazz The class to set up.
    '''

    if not issubclass(clazz, AxeCommandBase):
      msg = ('@BatchCommand-decorated command class %s ' +
             "must be a subclass of AxeCommandBase, but isn't.")
      msg = msg % clazz.__name__
      raise AxeException(msg)

    # Check that it has an init() and run() method.

    required = ['init', 'run', 'fini']
    missing = []
    for item in required:
      if not hasattr(clazz, item):
        missing.append(item)

    if missing:
      msg = 'Batch command %s is missing methods: %s'
      msg = msg % (clazz.__name__, ", ".join(missing))
      raise AxeException(msg)

    # Check that the text attributes are set.

    required = ['name', 'description', 'usage']
    missing = []
    for item in required:
      if getattr(clazz, item) == 'must be overridden':
        missing.append(item)

    if missing:
      msg = ('@BatchCommand-decorated command class %s ' +
             'is missing variables: %s')
      msg = msg % (clazz.__name__, ", ".join(missing))
      raise AxeException(msg)

    BatchCommand.registeredCommands.append(clazz)

    # Mark the class as a batch command.
    clazz.isBatchCommand = True

    return clazz

  #-----------------------------------------------------------------
  def __call__(self, clazz: Type[AxeCommandBase]) -> Type[AxeCommandBase]:
    '''Sets up a class to be a batch command.

    @param clazz The class to set up.
    '''

    return self.decorate(clazz)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
