#-------------------------------------------------------------------
#  AxeCommand.py
#
#  The AxeCommand class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the AxeCommand class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .AxeCommandBase import AxeCommandBase
from ai.axe.web.core import AxeException
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
from typing import Type
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class AxeCommand:
  '''Decorator that marks a class as an Axe command-line tool.
  '''

  #-----------------------------------------------------------------
  # Static list of all registered commands.
  registeredCommands: List[Type[AxeCommandBase]] = []

  #-----------------------------------------------------------------
  @staticmethod
  def find(commandName: str) -> Optional[Type[AxeCommandBase]]:
    '''Returns the command with the specified name, or None.
    '''

    for cmd in AxeCommand.registeredCommands:
      if cmd.name == commandName:
        return cmd

    return None

  #-----------------------------------------------------------------
  def __init__(self):
    '''Creates a new AxeCommand.
    '''

    pass

  #-----------------------------------------------------------------
  def decorate(self, clazz: Type[AxeCommandBase]) -> Type[AxeCommandBase]:
    '''Sets up a class to be executed as a command.

    @param clazz The class to set up.
    '''

    if not issubclass(clazz, AxeCommandBase):
      msg = ('@AxeCommand-decorated command class %s ' +
             "must be a subclass of AxeCommandBase, but isn't.")
      msg = msg % clazz.__name__
      raise AxeException(msg)

    # Check that it has an init() and run() method.

    required = ['run']
    missing = []
    for item in required:
      if not hasattr(clazz, item):
        missing.append(item)

    if missing:
      msg = 'Batch command %s is missing methods: %s'
      msg = msg % (clazz.__name__, ", ".join(missing))
      raise AxeException(msg)

    # Check that the text attributes are set.

    required = ['name', 'description', 'usage']
    missing = []
    for item in required:
      if getattr(clazz, item) == 'must be overridden':
        missing.append(item)

    if missing:
      msg = ('@AxeCommand-decorated command class %s ' +
             'is missing variables: %s')
      msg = msg % (clazz.__name__, ", ".join(missing))
      raise AxeException(msg)

    AxeCommand.registeredCommands.append(clazz)

    # Mark the class as a non-batch command.
    clazz.isBatchCommand = False

    return clazz

  #-----------------------------------------------------------------
  def __call__(self, clazz: Type[AxeCommandBase]) -> Type[AxeCommandBase]:
    '''Sets up a class to be a batch command.

    @param clazz The class to set up.
    '''

    return self.decorate(clazz)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
