#-------------------------------------------------------------------
#  AxeBrowserTestCase.py
#
#  The AxeBrowserTestCase class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the AxeBrowserTestCase class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.unittest.AxeSimpleTestCase import AxeSimpleTestCase
from selenium.webdriver.remote.webdriver import WebDriver
from subprocess import Popen
from typing import Optional
from typing import Type
import os
import selenium.webdriver
import time
import unittest
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class AxeBrowserTestCase(AxeSimpleTestCase):
  '''Unittest TestCase with AXE-specific functionality added.
  '''

  # The URL that all tests will connect to.
  url: str = ''

  # Whether all tests will be run in headless mode.
  isHeadless: bool = True

  #-----------------------------------------------------------------
  def __init__(self, *args, **kwargs) -> None:
    '''Creates a new AxeBrowserTestCase.
    '''

    AxeSimpleTestCase.__init__(self, *args, **kwargs)

    # The browsers that this test will be run with (i.e. Firefox, Chrome).
    self.browserClasses = [#selenium.webdriver.Firefox,
                           selenium.webdriver.Chrome]

    # The user name that the browser will be logged in with.
    self.logInUserName = 'mbrady'

    # The password that the browser will be logged in with.
    self.logInPassword = 'password'

    # The browser object to be used by the test.
    self.browser: Optional[WebDriver] = None

    # The browser class currently being used (i.e. Firefox, Chrome).
    self.browserClass: Optional[Type[WebDriver]] = None

    # The virtual X server process if we're running headless.
    self.xProcess: Optional[Popen] = None

  #-----------------------------------------------------------------
  def getBrowser(self) -> WebDriver:
    '''Returns a browser that is logged in and pointing at the project webapp.
    '''

    # Import Selenium here to avoid the dependency for non-browser projects.
    # pylint: disable=import-outside-toplevel
    from selenium.webdriver.common.keys import Keys

    browser = self.startBrowser()

    url = AxeBrowserTestCase.url
    url += '/'

    browser.get(url)

    userName = browser.find_element_by_name('userName')
    userName.send_keys(self.logInUserName)
    password = browser.find_element_by_name('password')
    password.send_keys(self.logInPassword)
    password.send_keys(Keys.RETURN)

    print("Waiting for login to finish...", end='')
    time.sleep(8)
    print("...continuing")

    return browser

  #-----------------------------------------------------------------
  def startBrowser(self) -> WebDriver:
    '''Returns a browser for use in testing.

    If no browser currently exists, creates a new one.

    This should be called within a unit test to get the browser.
    '''

    assert self.browserClass is not None

    if not self.browser:
      self.browser = self.browserClass()
      self.addCleanup(self.quitBrowser)

    return self.browser

  #-----------------------------------------------------------------
  def quitBrowser(self) -> None:

    if self.browser:
      self.browser.quit()
      self.browser = None

  #-----------------------------------------------------------------
  def run(self,   # type: ignore  # typeshed stubs have wrong return type.
          result: Optional[unittest.TestResult] = None) -> None:
    '''Called by JUnit to run a test.
    '''

    # Set up browser support, then call the superclass run() method.

    assert not self.browser

    try:
      if AxeBrowserTestCase.isHeadless:
        self.startX()

      try:

        for browserClass in self.browserClasses:
          self.browserClass = browserClass

          # Call the base class 'run' method.
          AxeSimpleTestCase.run(self, result)

      finally:
        self.quitBrowser()

    finally:
      self.stopX()

  #-----------------------------------------------------------------
  @staticmethod
  def setIsHeadless(isHeadless: bool) -> None:
    '''Sets whether all tests will be run in headless mode.
    '''

    AxeBrowserTestCase.isHeadless = isHeadless

  #-----------------------------------------------------------------
  @staticmethod
  def setUrl(url: str) -> None:
    '''Sets the URL that all tests will connect to.
    '''

    AxeBrowserTestCase.url = url

  #-----------------------------------------------------------------
  def startX(self) -> None:
    '''Starts a virtual X display for use in a headless environment.
    '''

    assert not self.xProcess, self.xProcess

    displayNumber = 3
    displayLabel = ':%d' % displayNumber

    xvfbLocations = ('/bin/Xvfb', '/usr/bin/Xvfb')
    xvfbLocation = None
    for location in xvfbLocations:
      if os.path.exists(location):
        xvfbLocation = location
        break

    assert xvfbLocation, xvfbLocation

    cmd = [xvfbLocation, displayLabel]

    self.xProcess = Popen(cmd)

    os.environ['DISPLAY'] = displayLabel

  #-----------------------------------------------------------------
  def stopX(self) -> None:
    '''Stops the server started with 'startX()'.
    '''

    if self.xProcess:
      self.xProcess.terminate()
      self.xProcess = None
