#-------------------------------------------------------------------
#  commands.py
#
#  The commands module.
#
#  Copyright 2013  Applied Minds, Inc
#-------------------------------------------------------------------

"""The commands used to deploy.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
from .DeploySetup import DeploySetup
from ai.axe.web.app import AppSetup
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
def deployCommands(installName: str,
                   version: str,
                   isRemote: bool,
                   remoteTag: Optional[str],
                   updateDb: bool,
                   installUser: str,
                   storeUser: str) -> List[str]:
  """Given an install, returns the commands to do a deploy there.

  @param installName The installation for which the config is wanted.
  @param version The version to be installed.  For example: 1.5
  @param isRemote True if we're copying from a remote server.  False
                  if we're copying from the local 'build' directory.
  @param remoteTag If isRemote is True, this tells the tag to copy from.
  @param upgradeDb If True, the DB schema will be upgraded to the current
                   version as part of the deploy.
  @param installUser The user name to use for ssh commands.
  @param storeUser The user name to use to log in to the tarball store server.
                   Not used if isRemote=False.
  """

  deploySetup = DeploySetup.get()
  host = deploySetup.site(installName)['host']
  platform = deploySetup.site(installName)['platform']

  appSetup = AppSetup.get()
  appName = appSetup.appName()
  appNameCap = appName[:1].upper() + appName[1:]

  installHost = host

  distDirName = appName + "-" + version
  distTarBaseName = distDirName + "-" + platform

  if installUser:
    sshHost = installUser + "@" + installHost
  else:
    sshHost = installHost

  # Prefix for ssh commands.
  ssh = "ssh " + sshHost + " "

  # The tarball to be copied.
  # If 'local', we'll copy from the 'build' directory.
  # If 'remote', we'll copy from the server.
  if isRemote:
    assert remoteTag is not None
    storeHost, storeDir = deploySetup.storeDir()
    if storeUser:
      storeHost = storeUser + '@' + storeHost
    fromDir = storeHost + ':' + storeDir + '/' + remoteTag
  else:
    fromDir = 'build/dist'

  tarGzBase = distTarBaseName + '.tar.gz'
  tarGzFile = fromDir + '/' + tarGzBase

  confFile = appName + '.conf'
  wsgiFile = 'wsgi' + appNameCap
  envFile = appName + 'Env.sh'
  appAdmin = appName + 'Admin'

  installRootDir = deploySetup.siteDir(installHost)
  installDirectory = installRootDir + "/" + installName
  installTarGz = installDirectory + "/" + tarGzBase
  installAppDir = installDirectory + "/" + distDirName
  installConfSrc = installAppDir + '/conf/' + installName  + '/' + confFile
  installConf = installAppDir + '/' + confFile
  installWsgi = installAppDir + '/bin/' + wsgiFile
  installSymlink = installDirectory + "/enabled"

  sedInstallDir = 'sed -i "s|{INSTALL_DIR}|' + installAppDir  + '|g" '

  # Commands to install the webapp.
  cmds = [
    ssh + "rm -f " + installTarGz,
    ssh + "rm -rf " + installAppDir,
    ssh + "mkdir -p " + installDirectory,
    #ssh + "chmod g+w " + installDirectory,

    "scp " + tarGzFile + " " + sshHost + ":" + installTarGz,
    ssh + "tar zxf " + installTarGz + " -C " + installDirectory,

    # Point the 'enabled' symlink at the newly copied install.
    ssh + "rm -f " + installSymlink,
    ssh + "ln -s " + installAppDir  + " " + installSymlink,

    # Put the correct .conf in place.
    ssh + "cp " + installConfSrc + " " + installConf,

    # Put the path to Python in the wsgi script.
    ssh + "'" + sedInstallDir + " " + installWsgi + "'",

    ssh + "chmod -R g+w " + installAppDir,
    ssh + "chmod g+w " + installTarGz,
  ]

  if updateDb:
    cmds += [
      # Update the database to the current version.
      ssh + ("cd " + installAppDir + "; " +
             ". ./" + envFile + "; " +
             appAdmin + " updateDb -run")
     ]

  return cmds
