#-------------------------------------------------------------------
#  DeploySetup.py
#
#  The DeploySetup class.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the DeploySetup class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .DotTemplate import DotTemplate
from ai.axe.web.app import AppSetup
from importlib import import_module
from types import ModuleType
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class DeploySetup:
  '''The setup for the Axe deploy system.

  How and where to deploy your app.

  This includes information about:

    - what goes in the config file, and how it varies from install to install.
    - pre-configured deploy targets (for example, dev, test, and production).

  To use the Axe build functionality, you must create a subclass
  of DeploySetup and register it using DeploySetup.register().

  This must be done before using any other Axe build functionality.
  '''

  #-----------------------------------------------------------------
  # The DeploySetup currently being used.
  setup: Optional['DeploySetup'] = None

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new DeploySetup.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def register(deploySetup: 'DeploySetup') -> None:
    '''Registers a deploySetup for the current Axe build.

    @param deploySetup An object that is a DeploySetup subclass.
    '''

    DeploySetup.setup = deploySetup

  #-----------------------------------------------------------------
  def confTemplate(self) -> str:
    '''Returns the string that is the .conf file template.

    Any "${variableName}" items found in the file will be filled
    in from the configDefaults() dictionary, or a deploySites()
    dictionary.

    @return A string that is the .conf file template.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def confDefaults(self) -> Dict[str, str]:
    '''Returns a dictionary that is the .conf file default values.

    These values will be used to fill in any "${keyName}" items
    in the configTemplate() string.

    A value found in a deploySites() dictionary will override this default.

    @return A {str: str} dictionary of {variableName: defaultValue}.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def sites(self) -> Dict[str, Dict[str, str]]:
    '''Returns a dictionary of .conf values for each site.

    These values will be used to fill in any "${keyName}" items
    in the configTemplate() string.

    Each site dictionary must have one additional key that is not used
    in the conf file:  web_host.  The web_host is the machine that
    the software will be deployed to.

    @return A {siteName: siteConfig} dictionary, where siteName is a string,
            and siteConfig is a {variableName: variableValue}} dictionary of
            type {str: str} where each variable will be set in the
            .conf file for that site.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def siteDirs(self) -> Dict[str, str]:
    '''Returns a dictionary of install directory per web host.

    This directory is where the app will be deployed to.

    @return A {siteName: installDirectory} dictionary,
             where siteName and installDirectory are strings.
             The special siteName of '*' will be a default directory
             for any web_host not explicitly named.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def storeDir(self) -> Tuple[str, str]:
    '''The server and directory where release tarballs are stored.

    This directory is where tarballs will be copied to after building,
    and will be copied from for deployment.

    @return A (server, directory) that can be used as an ssh source/desination.
    '''

    appName = AppSetup.get().appName()

    storeHost = appName + 'dev.appliedinvention.com'

    storeDir = '/var/' + appName + '-release/builds'

    return storeHost, storeDir

  #-----------------------------------------------------------------
  @staticmethod
  def get() -> 'DeploySetup':
    '''Returns the deploySetup for the current Axe app.

    @return An object that is a DeploySetup subclass.
    '''

    if DeploySetup.setup:
      return DeploySetup.setup
    else:
      msg = "Error.  The DeploySetup.register() method has not yet been called "
      msg += "to register the DeploySetup class for your Axe webapp."
      raise ValueError(msg)

  #-----------------------------------------------------------------
  def loadDirectory(self,
                    packageName: str,
                    siteNames: List[str]) -> List[Dict[str, str]]:
    '''Loads a directory full of site config files.

    @param packageName The package that contains the site files.
    @param siteNames List of site names.  For each name, a *.py file
                     will be imported.

    @param a list of {deploy_key, deploy_value} dictionaries, one per passed-in
           site name.
    '''

    modules: List[ModuleType] = self.loadDirectoryModules(packageName,
                                                          siteNames)

    dicts: List[Dict[str, str]] = []

    for module in modules:

      if not hasattr(module, 'install'):
        msg = 'Error:  module %s is missing an "install" dictionary.'
        msg = msg % module
        raise ValueError(msg)

      installObj = getattr(module, 'install')

      if not isinstance(installObj, dict):
        msg = 'Error:  module %s "install" member should be a dictionary, '
        msg += 'but is actually a %s'
        msg = msg % (module, type(installObj))
        raise ValueError(msg)

      installDict = installObj

      for key, value in installDict.items():

        if not (isinstance(key, str) and isinstance(value, str)):
          msg = 'Error:  module %s "install" dict has invalid key/value. '
          msg += 'They should be str/str, but are actually %s/%s.  '
          msg += 'Key/value are: %s:  %s'
          msg = msg % (module, type(key), type(value), key, value)
          raise ValueError(msg)

      dicts.append(installDict)

    return dicts

  #-----------------------------------------------------------------
  def loadDirectoryModules(self,
                    packageName: str,
                    siteNames: List[str]) -> List[ModuleType]:
    '''Loads a directory full of site config files.

    @param packageName The package that contains the site files.
    @param siteNames List of site names.  For each name, a *.py file
                     will be imported.

    @param a list of modules.
    '''

    modules: List[ModuleType] = []

    for siteName in siteNames:

      try:
        modules.append(import_module('.' + siteName, packageName))

      except ImportError as ex:
        msg = ("Error trying to load site config file " + siteName + ".py " +
               "from directory '" + packageName + "'.  Either add/fix the " +
               siteName + ".py file, or remove the '" + siteName+ "' site " +
               "from the list in the " + str(type(self)) + " class.\n" +
               "Import Error:  " + str(ex))
        raise ValueError(msg)

    return modules

  #-----------------------------------------------------------------
  def siteDir(self, webHost: str) -> str:
    '''Returns the install directory for the specified web host.

    @param webHost The web_host from the site dictionary.

    @param A string directory name.
    '''

    siteDirs = self.siteDirs()

    if webHost in siteDirs:
      return siteDirs[webHost]
    elif '*' in siteDirs:
      return siteDirs['*']
    else:
      msg = 'No install directory found for web host "%s".  ' % webHost
      msg += 'Check the siteDirs() method of class %s.' % type(self)
      raise ValueError(msg)

  #-----------------------------------------------------------------
  def site(self, siteName: str) -> Dict[str, str]:
    '''Returns the .conf file dictionary for the specified site.

    @param siteName The string name of a deploy site.

    @return A {str: str} site configuration dictionary.
    '''

    sitesDict = self.sites()

    if siteName not in sitesDict:
      msg = 'No deploy config found for site "%s".  ' % siteName
      msg += 'Check the sites() method of class %s.' % type(self)
      raise ValueError(msg)

    return sitesDict[siteName]

  #-----------------------------------------------------------------
  def siteConf(self, siteName: str) -> str:
    """Returns a string that is the contents of a config file .

    @param siteName The deploy site for which the config is wanted.
    @return The contents of the config file as a string.
    """

    installDict = dict(self.site(siteName))

    # Add default values.
    for key, defaultValue in self.confDefaults().items():
      installDict.setdefault(key, defaultValue)

    templ = DotTemplate(self.confTemplate())
    return templ.substitute(installDict)

  #-----------------------------------------------------------------
  def siteNames(self) -> List[str]:
    '''Returns a list of all the available site names.
    '''

    return list(self.sites().keys())
