#-------------------------------------------------------------------
#  WebConstruct.py
#
#  The WebConstruct module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''This module is an SCons SConstruct file.

Import this file *only* if you want to set up SCons to build your project.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from SCons.Environment import Base as SconsEnvironment
from SCons.Node import Node
from SCons.Node import NodeList
from SCons.Script import ARGUMENTS
from ai.axe.build import BuildPlatform
from ai.axe.build import BuildSetup
from ai.axe.build.sconstruct.Construct import Construct
from ai.axe.build.sconstruct.Util import recursiveGlob
from ai.axe.build.sconstruct.scons.TarGz import tarCmd
from ai.axe.build.sconstruct.scons.Util import newSectionProgress
from ai.axe.web.app import AppSetup
from ai.axe.web.build.sconstruct.installations import DeploySetup
from compileall import compile_dir
from glob import glob
from typing import Dict
from typing import List
from typing import Optional
import ai.axe.build.sconstruct.Coverage
import ai.axe.build.sconstruct.scons.Copy
import ai.axe.build.sconstruct.scons.Defaults
import ai.axe.build.sconstruct.scons.PyTest
import ai.axe.build.sconstruct.scons.TarGz
import ai.axe.build.sconstruct.scons.TsLint
import ai.axe.build.sconstruct.scons.Tsc
import ai.axe.web.build.Typescript
import ai.axe.web.build.sconstruct.installations
import importlib
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WebConstruct(Construct):
  '''An SCons construction setup.
  '''

  #-------------------------------------------------------------------
  def __init__(self,
               buildSetup: BuildSetup,
               deploySetup: DeploySetup):
    '''Creates a new WebConstruct object.
    '''

    Construct.__init__(self, buildSetup)

    # The deploy setup being used.
    self.deploySetup: DeploySetup = deploySetup

    # The distibution bundle directory name.
    self.distDirName = self.appName + "-" + self.versionFile.versionName()

    # The distibution bundle tar file base name.
    self.distTarBaseName = self.distDirName + "-" + BuildPlatform.platform()

    # The list of files in the distribution bundle.
    self.distFiles: NodeList = NodeList()

    # The distribution bundle (unzipped) tar file.
    self.distTar: Optional[NodeList] = None

  #-----------------------------------------------------------------
  def addDistFiles(self,
                   files: List[str],
                   srcDir: str,
                   destDir: str,
                   noLink: bool = False) -> NodeList:
    """Adds files to the distribution tarball.

    @param files The source files, relative to {project}_HOME.
    @param srcDir The source dir, relative to {project}_HOME.
    @param destDir The dest dir, relative to the distribution root.
    @param noLink If true, a symlink encountered will be replaced by its
                  link target file.

    @return A list of file nodes.
    """

    nodes: NodeList = self.addFilesToBundle(files, srcDir, destDir, noLink)

    self.env.Depends(self.distTar, nodes)

  #-------------------------------------------------------------------
  @staticmethod
  def createWeb(buildSetup: BuildSetup,
               deploySetup: DeploySetup) -> 'WebConstruct':
    '''Run this file to set up SCons to build your file.
    '''

    BuildSetup.register(buildSetup)
    DeploySetup.register(deploySetup)

    construct = WebConstruct(buildSetup, deploySetup)

    # Set up the basic build functionality.
    construct.addTargets()

    # Add webapp-specific build functionality.
    construct.addWebTargets()

    return construct

  #-----------------------------------------------------------------
  def addWebTargets(self) -> None:
    '''Add targets to this Construct object.
    '''

    aenv = self.env

    appSetup: AppSetup = AppSetup.get()
    buildSetup: BuildSetup = BuildSetup.get()
    deploySetup: DeploySetup = DeploySetup.get()

    #----------------------------------------------------------------------
    appName = self.appName
    appNameCap = self.appNameCap

    #----------------------------------------------------------------------
    def addTarget(name: str, description: str, isMajor: bool = False):
      self.addTarget(name, description, isMajor)

    # Set up the typescript builder.
    ai.axe.build.sconstruct.scons.Tsc.addTscBuilder(aenv)

    # The typescript compiler.
    aenv['NODE'] = self.toolDir + '/bin/node'
    aenv['TSC'] = self.toolDir + '/bin/tsc'
    aenv['TSLINT'] = self.toolDir + '/bin/tslint'

    # The Sass compiler.
    aenv['SASS'] = self.toolDir + '/dart-sass/sass'

    rootPackageDir = self.rootPackageDir

    # The config file name.
    configFile = appName + '.conf'

    # Binaries needed to run the web app.
    toolBinDir = self.toolDir + '/bin'

    # Shared libraries needed to run the web app.
    libDir = self.toolDir + '/lib'

    # Python libraries needed to run the web app.
    appPylibDir = self.toolDir + '/pylib/lib/python3.9/site-packages'
    appPylib64Dir = self.toolDir + '/pylib/lib64/python3.9/site-packages'

    # Python libraries needed to run standalone.
    #cmdPylibDir = toolDir + '/pylib-cmd'

    # Python libraries needed to build the app.
    #devPylibDir = toolDir + '/pylib-dev'

    #----------------------------------------------------------------------
    # Rules to run the browser tests.

    newSectionProgress('browser test')

    # Whether tests that require X should launch their own X server.
    isHeadlessStr = ARGUMENTS.get('headless', 'false')
    isHeadless = (isHeadlessStr == 'true')
    aenv['isHeadless'] = isHeadless

    # The server that browser tests should connect to.
    browserTestServer = ARGUMENTS.get('browserTestServer', 'local')

    # Build the URL of the browser test server.

    serverConfigs: Dict[str, Dict[str, str]] = deploySetup.sites()
    validServers = list(serverConfigs.keys()) + ['local']
    assert browserTestServer in validServers, (browserTestServer, validServers)

    if browserTestServer == 'local':
      serverUrl: str = 'http://localhost:%s' % buildSetup.localDevServerPort()
    else:
      serverUrl = 'https://%s' % validServers[browserTestServer]['web_host']
    aenv['browserTestUrl'] = serverUrl

    ai.axe.build.sconstruct.scons.PyTest.addPyTestBuilder(aenv)

    testSrcFiles = recursiveGlob('src/browserTest', 'Test*.py')
    testSrcFiles.sort()

    for testSrcFile in testSrcFiles:

      # Make the path relative to the build directory.
      relName = testSrcFile[len('src/browserTest/'):]
      testClassName = relName.split('/')[-1]
      testClassName = testClassName[:-len('.py')]

      testPassedName = 'build/browserTest/' + relName
      testPassedName = testPassedName[:-len('.py')]
      testPassedName += '.passed'

      # pylint: disable=E1101
      aenv.PyTest(testPassedName, testSrcFile)
      # pylint: enable=E1101

      # Browser tests require a build ID to be in place.
      aenv.Depends(testPassedName, self.buildIdFile.fileName())

      aenv.Alias('browserTest', testPassedName)
      aenv.Alias('browserTest-' + testClassName, testPassedName)
      addTarget('browserTest', 'Run all browser tests.')
      addTarget('browserTest-' + testClassName,
                'Run the browser test for class ' + testClassName)

      # Until dependencies are working, always re-run browser tests.
      aenv.AlwaysBuild(testPassedName)

    #----------------------------------------------------------------------
    # Rules to build the APP.conf files for various deploy targets.

    newSectionProgress(configFile)

    axeConfigFiles = {}

    for aDeployName in deploySetup.siteNames():

      axeConfigFile = ("build/conf/" + aDeployName + "/" + configFile)
      axeConfigFiles[aDeployName] = axeConfigFile

      action = aenv.Action(self.writeConfigFile, self.writeConfigFileMessage)

      aenv.Command(axeConfigFile, '', action)

    #----------------------------------------------------------------------
    # Rules to build the axe.conf file for the development environment.

    devAxeConfigFile = "build/" + configFile
    devAxeConfig = self.axeConfCommand(devAxeConfigFile, 'dev')
    aenv.Alias('devConf', devAxeConfig)

    addTarget('devConf', 'The %s file for the dev environment.' % configFile)

    ###----------------------------------------------------------------------
    ### Rules to build the HTML files.
    ##
    ##newSectionProgress('axe.html')
    ##
    ##jsSrcFiles = Compress.readJsSourceListFile('src/webapp/axeIncludes.txt')
    ##jsSrcFiles = [ ('src/webapp/' + f) for f in jsSrcFiles ]
    ##cssSrcFiles = Compress.readCssSourceListFile('src/webapp/axeIncludes.css')
    ##cssSrcFiles = [ ('src/webapp/' + f) for f in cssSrcFiles ]
    ##
    ##def writeDebugHtmlFile(target, source, env):
    ##
    ##  text = Compress.genDebugHtml('src/webapp', 'axe.html',
    ##                               'axeIncludes.txt', 'extIncludes.txt')
    ##  open(str(target[0]), 'w').write(text.encode('utf-8'))
    ##
    ##msg = 'Generating debug axe.html...'
    ##action = aenv.Action(writeDebugHtmlFile, msg)
    ##axeDebug = aenv.Command('build/html/debug/axe.html', jsSrcFiles, action)
    ##
    ##def writeReleaseHtmlFile(target, source, env):
    ##
    ##  text = Compress.genReleaseHtml('src/webapp', 'axe.html', 'axe.js',
    ##                                 'ext.js')
    ##  open(str(target[0]), 'w').write(text.encode('utf-8'))
    ##
    ##msg = 'Generating release axe.html...'
    ##action = aenv.Action(writeReleaseHtmlFile, msg)
    ##axeRelease = aenv.Command('build/html/release/axe.html', jsSrcFiles, action)
    ##
    ##def writeReleaseJsFile(target, source, env):
    ##
    ##  targetName = os.path.basename(str(target[0]))
    ##  text, sourceMapText, fileNames = Compress.genJsFile('src/webapp',
    ##                                                      'axeIncludes.txt',
    ##                                                      targetName)
    ##  open(str(target[0]), 'w').write(text.encode('utf-8'))
    ##  open(str(target[0]) + '.map', 'w').write(sourceMapText.encode('utf-8'))
    ##
    ##msg = 'Generating release axe.js...'
    ##action = aenv.Action(writeReleaseJsFile, msg)
    ##jsRelease = aenv.Command('build/html/release/axe.js', jsSrcFiles, action)
    ##
    ##def writeReleaseExtJsFile(target, source, env):
    ##
    ##  targetName = os.path.basename(str(target[0]))
    ##  text, sourceMapText, fileNames = Compress.genJsFile('src/webapp',
    ##                                                      'extIncludes.txt',
    ##                                                      targetName, appJsDir)
    ##  open(str(target[0]), 'w').write(text.encode('utf-8'))
    ##  open(str(target[0]) + '.map', 'w').write(sourceMapText.encode('utf-8'))
    ##
    ##msg = 'Generating release ext.js...'
    ##action = aenv.Action(writeReleaseExtJsFile, msg)
    ##src = 'src/webapp/extIncludes.txt'
    ##extJsRelease = aenv.Command('build/html/release/ext.js', src, action)
    ##
    ##def writeReleaseCssFile(target, source, env):
    ##
    ##  targetName = os.path.basename(str(target[0]))
    ##  text, sourceMapText, fileNames = Compress.genCssFile('src/webapp',
    ##                                                       'axeIncludes.css',
    ##                                                       targetName)
    ##  open(str(target[0]), 'w').write(text.encode('utf-8'))
    ##  open(str(target[0]) + '.map', 'w').write(sourceMapText.encode('utf-8'))
    ##
    ##msg = 'Generating release axe.css...'
    ##action = aenv.Action(writeReleaseCssFile, msg)
    ##jsRelease = aenv.Command('build/html/release/axe.css', cssSrcFiles, action)

    ###----------------------------------------------------------------------
    ### Rules to lint the Javascript code.
    ##
    ##newSectionProgress('jslint')
    ##
    ##text = open('src/webapp/axeIncludes.txt').read()
    ##lines = text.strip().split('\n')
    ##
    ### Strip comments.
    ##for i, line in enumerate(lines):
    ##  if line.find('#') > -1:
    ##    lines[i] = line[:-line.index('#')]
    ##
    ### Strip empty lines.
    ##lines = [ x for x in lines if x.strip() ]
    ##
    ### Rule to create the config file.
    ##
    ##classNames = []
    ##for fileName in lines:
    ##  justFileName = os.path.split(fileName)[1]
    ##  className = os.path.splitext(justFileName)[0]
    ##
    ##  ignoreNames = ('Axe', 'axeBootstrap', 'array', 'assert', 'data', 'date',
    ##                 'func', 'gui', 'http', 'scope', 'string')
    ##
    ##  if className not in ignoreNames:
    ##    classNames.append(className)
    ##
    ##classNamesText = '"' + '": false, \n    "'.join(classNames) + '": false'
    ##
    ##def createJslintConfig(target, source, env):
    ##  text = open(jshintConfigSrc).read()
    ##  text = text.replace('ADD_MORE_GLOBALS_HERE', classNamesText)
    ##  #text = text.replace('ADD_MORE_GLOBALS_HERE','"foo":false')
    ##  open(jshintConfig, 'w').write(text)
    ##
    ##msg = 'Creating jshintrc...'
    ##action = aenv.Action(createJslintConfig, msg)
    ##cmd = aenv.Command(jshintConfig, jshintConfigSrc, action)
    ##
    ### Rules to lint each file.
    ##
    ##for fileName in lines:
    ##
    ##  fileName = 'src/webapp/' + fileName
    ##
    ##  outFile = 'build/jslint/' + fileName + '.passed'
    ##
    ##  cmds = [ jshint + ' ' + fileName,
    ##           'cp ' + fileName + ' ' + outFile ]
    ##  msg = 'Linting file %s...' % fileName
    ##  action = aenv.Action(cmds, msg)
    ##  cmd = aenv.Command(outFile, [fileName, jshintConfig], action)
    ##
    ##  justFileName = os.path.split(fileName)[1]
    ##
    ##  aenv.Alias('jslint-' + justFileName, cmd)
    ##  aenv.Alias('jslint', cmd)
    ##
    ###aenv.Alias('pylint', 'build/pylint/axe.passed')

    #----------------------------------------------------------------------
    # The generated web action Typescript file.

    webActionTsFiles = [
      'src/webapp/htmllib/gen/classJsonInterface.ts',
      'src/webapp/htmllib/gen/classJsonClass.ts',
      'src/webapp/htmllib/gen/webActions.ts',
      ]

    # All action python files are inputs to this command.
    actionRoot = rootPackageDir + '/actions'
    actionSrc = glob(actionRoot + '/**/*.py')

    msg = 'Generating web action API "' + webActionTsFiles[0] + '"...'
    action = aenv.Action(self.generateWebActionTs, msg)

    webActionTs = aenv.Command(webActionTsFiles, actionSrc, action)

    aenv.Alias('webActionTs', webActionTs)

    addTarget('webActionTs', 'Generate the web action Typescript file.')

    #----------------------------------------------------------------------
    # The Typescript files compiled into APP.js.

    if appSetup.hasFrontEnd():

      # Copy the Axe TS files into the source tree now, so the file
      # names are available to find to create the Typescript builder.
      # This is bad because if there's an error here, it will
      # break all SCons commands, but I can't think of anything better.
      ai.axe.web.build.Typescript.copyAxeTsFiles()

    tsSrcFiles = ai.axe.web.build.Typescript.findTsFiles(False)
    tsTestFiles = ai.axe.web.build.Typescript.findTsFiles(True)

    tsSrcFiles.extend(webActionTsFiles)
    tsTestFiles.extend(webActionTsFiles)

    axeJsFile = '%s.js' % appName
    axeJsTestFile = '%s-test.js' % appName

    # pylint: disable=E1101
    axeJs = aenv.Tsc('build/html/' + axeJsFile, tsSrcFiles)
    axeJsTest = aenv.Tsc('build/html/' + axeJsTestFile, tsTestFiles)
    # pylint: enable=E1101

    axeJsLabel = appName + 'Js'
    axeJsTestLabel = appName + 'JsTest'

    aenv.Alias(axeJsLabel, axeJs)
    aenv.Alias(axeJsTestLabel, axeJsTest)

    addTarget(axeJsLabel, 'Compile Typescript files.')
    addTarget(axeJsTestLabel, 'Compile Typescript files, including tests.')

    #----------------------------------------------------------------------
    # Typescript lint

    tsSrcFiles = ai.axe.web.build.Typescript.findTsFiles(True)

    # pylint: disable=E1101
    tsLint = aenv.TsLint('build/html/%sJsTsLintPassed' % appName, tsSrcFiles)
    # pylint: enable=E1101

    aenv.Depends(tsLint, axeJsTest)
    aenv.Alias('tslint', tsLint)

    addTarget('tsLint', 'Lint the Typescript files.')

    #----------------------------------------------------------------------
    # Theme scss file

    srcScss = 'src/webapp/themes.scss'
    destCss = 'build/html/themes.css'

    cmd = '$SASS ' + srcScss + ' ' + destCss

    action = aenv.Action(cmd, 'Compiling ' + destCss)

    aenv.Command(destCss, srcScss, action)

    #----------------------------------------------------------------------
    # Rules to build the distribution bundle.

    newSectionProgress('dist')

    distFiles = self.distFiles

    # Copy the webapp files.

    if appSetup.hasFrontEnd():

      # Make a list of files from src/webapp.
      files = recursiveGlob('src/webapp', '*')

      # Remove directories. They will be created when their contents are copied.
      for f in files[:]:
        if os.path.isdir(f):
          files.remove(f)

      distFiles += self.addFilesToBundle(files, "src/webapp", "webapp")

      # Generated HTML/JS files.
      files = ['build/html/%s.js' % appName,
               'build/html/%s.js.map' % appName,
               'build/html/themes.css',
               ]
      distFiles += self.addFilesToBundle(files, "build/html", "webapp")

    # The startup script.
    scriptName = 'admin/wsgi/wsgi' + appNameCap
    distFiles += self.addFilesToBundle([scriptName], "admin/wsgi", "bin")

    # The admin script.
    distFiles += self.addFilesToBundle(['admin/script/' + appName + 'Admin'],
                                  "admin/script", "bin")
    distFiles += self.addFilesToBundle(['admin/release/' + appName + 'Env.sh'],
                                  "admin/release", ".")

    # The config files.
    for fileName in axeConfigFiles.values():
      distFiles += self.addFilesToBundle([fileName], "build/conf", "conf")

    # Copy the SQL scripts.
    distFiles += self.addDirToBundle('admin/sql', "sql")

    # Copy the shared libraries.
    distFiles += self.addDirToBundle(libDir, "lib", noLink=True)

    # Make sure all .pyc file generated now, to avoid permission problems
    # generating them at run time.
    # Should really figure out to do this at build time, but for now,
    # just do it at SConstruct parse time.
    compile_dir('src/python', maxlevels=20, quiet=1)

    # Copy the Python libs.
    distFiles += self.addDirToBundle('src/python', "pylib/lib")

    # Copy the Python binary.
    distFiles += self.addDirToBundle(self.toolDir + '/python3.9', "python3.9")
    distFiles += self.addFilesToBundle([toolBinDir + '/python3'],
                                       toolBinDir, 'bin')

    # Copy the standalone Python libs.
    # Use 'addTree' so that hidden directories are included.
    distFiles += self.addTreeToBundle(appPylibDir, "pylib/lib")
    distFiles += self.addTreeToBundle(appPylib64Dir, "pylib/lib64")


    self.distTar = tarCmd(aenv,
                          "build/dist/" + self.distTarBaseName,
                          "build/dist/" + self.distDirName,
                          distFiles)

    # pylint: disable=E1101
    distTarGz = aenv.Gzip(self.distTar)
    # pylint: enable=E1101

    aenv.Alias('dist', distTarGz)
    addTarget('dist', 'Build a distribution bundle.')

    #----------------------------------------------------------------------
    # Add the 'scons targets' text for the targets in this file.

    targets = (
      ('tslint', 'Lint all typescript files.'),
      ('devConf', 'Create a dev enviroment ' + configFile + ' file.'),
      ('dist', 'Build a release distribution bundle.'),
      )

    for name, description in targets:
      addTarget(name, description, isMajor=True)


  #-----------------------------------------------------------------
  def writeConfigFile(self,
                      target: List[Node],
                      source: List[Node],
                      env: SconsEnvironment) -> None:
    '''Private helper method.  Action to write a config file.
    '''

    # The enclosing directory is the deployName.
    aDeployName = os.path.basename(os.path.dirname(str(target[0])))

    configContents = self.deploySetup.siteConf(aDeployName)
    open(str(target[0]), 'w', encoding='utf-8').write(configContents)

    # Suppress unused variable warnings.
    del source
    del env

  #-----------------------------------------------------------------
  def axeConfCommand(self, fileName: str, deployName: str) -> NodeList:

    devConfigContents = self.deploySetup.siteConf(deployName)

    # Note: this function is a closure using var devConfigContents.
    def writeDevConfigFile(target: List[Node],
                           source: List[Node],
                           env: SconsEnvironment) -> None:
      open(str(target[0]), 'w', encoding='utf-8').write(devConfigContents)

      # Supress unused variable warnings.
      # pylint: disable=self-assigning-variable
      source = source
      env = env

    action = self.env.Action(writeDevConfigFile,
                             self.writeConfigFileMessage)

    return self.env.Command(fileName, '', action)

  #-----------------------------------------------------------------
  def writeConfigFileMessage(self,
                             target: List[Node],
                             source: List[Node],
                             env: SconsEnvironment) -> str:
    '''Private helper method.  Writes a message for an Action.
    '''

    # Suppress unsed variable warnings.
    del source
    del env

    return "Writing config file " + str(target[0]) + "..."

  #-----------------------------------------------------------------
  def generateWebActionTs(self,
                          target: List[Node],
                          source: List[Node],
                          env: SconsEnvironment) -> None:
    '''Action adapter to write the web action .ts file.

    Private helper method.
    '''

    # Import in this function instead of in the SConstruct file
    # because this will initialize the action registry.
    # This would cause an error if done too early.

    # pylint: disable=import-outside-toplevel

    importlib.import_module(self.rootPackage + '.actions')
    from ai.axe.web.build.WebActionTs import WebActionTs

    WebActionTs.generate(str(target[0].Dir('.')))

    # Suppress unused variable warning.
    del source
    del env

  #-----------------------------------------------------------------
  def addDirToBundle(self,
                     srcDir: str,
                     destDir: str,
                     noLink: bool = False) -> NodeList:
    """Private function to add all files in a directory to the distribution.

    @param srcDir The source dir, relative to AXE_HOME.
    @param destDir The dest dir, relative to the distribution root.
    @param noLink If true, a symlink encountered will be replaced by its
                  link target file.

    @return A list of file nodes.
    """

    files = glob(os.path.join(srcDir, "*"))
    files = sorted(files)

    return self.addFilesToBundle(files, srcDir, destDir, noLink)

  #-----------------------------------------------------------------
  def addFilesToBundle(self,
                       files: List[str],
                       srcDir: str,
                       destDir: str,
                       noLink: bool = False) -> NodeList:
    """Private function to add a list of files to the distribution.

    @param files The source files, relative to AXE_HOME.
    @param srcDir The source dir, relative to AXE_HOME.
    @param destDir The dest dir, relative to the distribution root.
    @param noLink If true, a symlink encountered will be replaced by its
                  link target file.

    @return A list of file nodes.
    """

    # This replaces the built-in Copy function with a function that does
    # the same thing but doesn't write a status message.
    copy = ai.axe.build.sconstruct.scons.Copy.action

    # Copy link targets (don't preserve links).
    copyNoLink = ai.axe.build.sconstruct.scons.Copy.noLinkAction

    # Which copy function action to use.
    if noLink:
      copy = copyNoLink

    if not srcDir.endswith("/"):
      srcDir += "/"

    distFiles = NodeList()
    for f in files:
      src = f
      assert f.startswith(srcDir), "File %s should start with %s" % (f, srcDir)
      baseName = f[len(srcDir):]

      dest = os.path.join("build/dist", self.distDirName, destDir, baseName)

      cmd = self.env.Command(dest, src, copy(dest, src))
      distFiles.append(cmd)
    return distFiles

  #-----------------------------------------------------------------
  def addTreeToBundle(self,
                      srcDir: str,
                      destDir: str) -> NodeList:
    """Private function to add a single directory to the distribution.

    This does a single copy of the directory without doing any dependency
    checking on children.  It should only be used for directories whose
    contents never change (like external tools).  It should not be used
    for project files which are edited.

    @param srcDir The source dir, relative to AXE_HOME.
    @param destDir The dest dir, relative to the distribution root.

    @return A list of file nodes.
    """

    # Copy a directory tree.
    copyTree = ai.axe.build.sconstruct.scons.Copy.treeAction

    dest = os.path.join("build/dist", self.distDirName, destDir)

    cmd = self.env.Command(dest, srcDir, copyTree(dest, srcDir))

    distFiles = NodeList()
    distFiles.append(cmd)
    return distFiles
