#-------------------------------------------------------------------
#  Typescript.py
#
#  The Typescript class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the Typescript class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import codecs
import json
import os
import shutil
from glob import glob
from ai.axe.util import FileUtil
from ai.axe.util import StringUtil
from ai.axe.web.app import AppSetup
from subprocess import Popen
from tempfile import NamedTemporaryFile
from tempfile import TemporaryDirectory
from typing import List
from .WebActionTs import WebActionTs
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class Typescript:
  '''Compiles typescript files.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new Typescript.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def compileAppJs(includeTests: bool) -> None:
    '''Compiles the APP.js file.

    @param includeTests Whether the unit tests should be compiled into the file.
                        If True, the tests will be included.
                        If False, only the core code will be included.
    '''

    appSetup = AppSetup.get()

    homeDir = os.environ[appSetup.appNameAllCaps() + '_HOME']
    outFileNameOnly = appSetup.appName() + '.js'
    outFileName = homeDir + '/build/webapp/' + outFileNameOnly

    print(' * Compiling ' + outFileNameOnly)

    print('   + Compiling theme.css...')
    wasError = Typescript.compileThemeCss(outFileName)

    if not wasError:

      WebActionTs.generateWebActionTs()

      print('   + Copying Axe typescript files locally...')
      Typescript.copyAxeTsFiles()

      sourceFileNames = Typescript.findTsFiles(includeTests)

      print('   + Compiling typescript...')
      wasError = Typescript.compileTs(outFileName, sourceFileNames)

      if not wasError:
        print('   + Packing javascript...')
        Typescript.pack(outFileName)

    print(' * ' + outFileNameOnly + ' compiled.')

  #-----------------------------------------------------------------
  @staticmethod
  def findTsFiles(includeTests: bool) -> List[str]:
    '''Returns a list of all .ts source files.

    @param includeTests  If True, test files will be included.  If False,
                         they will not.

    @return A list of file path strings.
    '''

    appSetup = AppSetup.get()
    homeDir = os.environ[appSetup.appNameAllCaps() + '_HOME']

    srcRoot = homeDir + "/src/webapp/htmllib"

    if includeTests:
      regExSuffix = "/**/*.ts"
    else:
      regExSuffix = "/**/!test/*.ts"

    srcPath = srcRoot + regExSuffix

    fileNames = FileUtil.globEx(srcPath)
    fileNames = sorted(fileNames)

    return fileNames

  #-----------------------------------------------------------------
  @staticmethod
  def copyAxeTsFiles() -> None:
    '''Copies Axe TS files into the local source tree.
    '''

    axeRoot: str = Typescript.axePath()

    dest = 'src/webapp/htmllib/axe'

    # Delete existing files.

    childDirs = glob(dest + '/*')
    for childDir in childDirs:
      if os.path.isdir(childDir):
        shutil.rmtree(childDir)

    # Copy new files.

    childDirs = glob(axeRoot + '/axe/*')

    for childDir in childDirs:
      if os.path.isdir(childDir):
        dirNameOnly = os.path.split(childDir)[1]
        destDir = dest + '/' + dirNameOnly
        shutil.copytree(childDir, destDir)

  #-----------------------------------------------------------------
  @staticmethod
  def findScssFiles() -> List[str]:
    '''Returns a list of all .scss source files.

    @return A list of file path strings.
    '''

    appSetup = AppSetup.get()
    homeDir = os.environ[appSetup.appNameAllCaps() + '_HOME']

    return [homeDir + "/src/webapp/themes.scss"]

  #-----------------------------------------------------------------
  @staticmethod
  def compileThemeCss(outJsFileName: str) -> int:
    '''Compiles .scss files to .css files.

    @param outJsFileName Name of the final packed .js file that will be written.
                         Any error message while CSS is being compiled will
                         be written to this file.
    '''

    appSetup = AppSetup.get()

    toolDir = os.environ[appSetup.appNameAllCaps() + '_TOOLS']

    sass = os.path.join(toolDir, 'dart-sass', 'sass')

    homeDir = os.environ[appSetup.appNameAllCaps() + '_HOME']
    outFileName = homeDir + '/build/webapp/themes.css'
    inFileName = homeDir + '/src/webapp/themes.scss'

    sassCmd = [sass, inFileName, outFileName]

    wasError = Typescript.runCmd(sassCmd, outJsFileName)
    if wasError:
      return 1

    return 0

  #-----------------------------------------------------------------
  @staticmethod
  def compileTs(outFileName: str, sourceFileNames: List[str]) -> int:
    '''Compiles .ts files to .js files.

    @param outFileName Name of the final packed .js file that will be written.
    @param sourceFileNames The names of the files to compile.
    '''

    appSetup = AppSetup.get()
    appName = appSetup.appName()

    toolDir = os.environ[appSetup.appNameAllCaps() + '_TOOLS']

    tsc = os.path.join(toolDir, 'bin', 'tsc')

    homeDir = os.environ[appSetup.appNameAllCaps() + '_HOME']
    outDirName = homeDir + '/build/js'
    rootDirRelative = 'src/webapp/htmllib'
    mainFileRelative = (rootDirRelative + '/' +
                        appName + '/' + appName + 'Main.ts')
    rootDir = homeDir + '/' + rootDirRelative
    mainFile = homeDir + '/' + mainFileRelative

    # Make sure source files are absolute.
    rawFileNames = sourceFileNames[:]
    for i, rawFileName in enumerate(rawFileNames):
      if not rawFileName.startswith('/'):
        sourceFileNames[i] = homeDir + '/' + rawFileName

    # The directory that contains all typescript headers.
    baseUrl = toolDir + "/tsinclude"

    assert (mainFile in sourceFileNames or
            mainFileRelative in sourceFileNames), (mainFile, sourceFileNames)

    tsConfigDict = {
      'compilerOptions': {
        "target": "ES6",
        "module": "commonjs",
        "noImplicitAny": True,
        "pretty": False,
        "experimentalDecorators": True,
        "emitDecoratorMetadata": True,
        "baseUrl": baseUrl,
        "outDir": outDirName,
        "rootDir": rootDir,
        "sourceMap": True,
        "incremental": True,
        "tsBuildInfoFile": homeDir + '/build/tsBuildCache'
        },
      'files': sourceFileNames,
    }

    # Write the tsconf conf temp file.
    # tsc requires the file to be called 'tsconfig.json', so we create
    # a temp directoy and put the file in that.

    with TemporaryDirectory() as tsConfigDirName:
      tsConfigFileName = tsConfigDirName + '/tsconfig.json'
      open(tsConfigFileName, 'w').write(json.dumps(tsConfigDict, indent=2))

      tscCmd = [tsc, '--build', tsConfigFileName]

      wasError = Typescript.runCmd(tscCmd, outFileName)

    if wasError:
      return 1

    return 0

  #-----------------------------------------------------------------
  @staticmethod
  def axePath() -> str:
    '''Returns the path to the Axe Typescript source files.
    '''

    appSetup = AppSetup.get()
    toolDir = os.environ[appSetup.appNameAllCaps() + '_TOOLS']

    axePath = toolDir + "/axe-ts"

    # If there's a local Axe dev area, use that instead of the tools version.
    if 'AXE_DEV_HOME' in os.environ and os.environ['AXE_DEV_HOME']:
      axePath = os.environ['AXE_DEV_HOME'] + '/src/webapp/htmllib'

    return axePath

  #-----------------------------------------------------------------
  @staticmethod
  def pack(outFileName: str) -> int:
    '''Packs many js files into a single .js file.

    @param outFileName The name of the file to write.
    '''

    appSetup = AppSetup.get()
    appName = appSetup.appName()

    toolDir = os.environ[appSetup.appNameAllCaps() + '_TOOLS']
    homeDir = os.environ[appSetup.appNameAllCaps() + '_HOME']

    webpack = os.path.join(toolDir, 'bin', 'webpack')

    outFilePath, outFileFile = os.path.split(outFileName)

    # Webpack's output path must be absolute, so make it absolute if it is not.
    if not os.path.isabs(outFilePath):
      outFilePath = os.path.join(homeDir, outFilePath)

    # Generate the contents of the webpack.conf file.

    webpackConf = {
      "mode": "development",
      "entry": "./build/js/%s/%sMain.js" % (appName, appName),
      "output": {
        "path": outFilePath,
        "filename": outFileFile,
      },
      "resolve": {
        "modules": [
          toolDir + "/jslib-app",
        ],
      },
      "devtool": "source-map",
    }

    webpackConfStr = 'module.exports = ' + json.dumps(webpackConf, indent=2)

    # Write the webpack conf temp file.

    webpackConfFile = NamedTemporaryFile()
    open(webpackConfFile.name, 'w').write(webpackConfStr)

    webpackCmd = [webpack,
                  '--config', webpackConfFile.name,
    ]

    wasError = Typescript.runCmd(webpackCmd, outFileName)
    if wasError:
      return 1

    return 0

  #-------------------------------------------------------------------
  @staticmethod
  def runCmd(cmd: List[str], errorJsFileName: str) -> bool:
    """Runs the specified command.

    @param cmd An array of strings to be executed.
    @param errorJsFileName The JS out file that should be written if there
                           is an error.

    @return True if there was an error, False if the cmd was a success.
    """

    # Need to use actual temp files, because subprocess.PIPE will hang
    # with large output.
    outTempFile = NamedTemporaryFile()
    errTempFile = NamedTemporaryFile()

    proc = Popen(cmd, stdout=outTempFile, stderr=errTempFile)
    proc.wait()

    failed = proc.returncode != 0

    if failed:

      # Make the output a javascript file that will display the error message
      # in the web browser debugger.

      outTempFile.flush()
      errTempFile.flush()
      out = open(outTempFile.name, 'r').read()
      err = open(errTempFile.name, 'r').read()

      errorText = Typescript.getErrorMsgJs(out + err)
      codecs.open(errorJsFileName, 'w', 'utf-8').write(errorText)

      # Write the error to stdout for the user to see
      # in the scons or runServer console.

      print("Error compiling Typescript:")
      print(out + err)

    return failed

  #----------------------------------------------------------------
  @staticmethod
  def getErrorMsgJs(errorMsg: str) -> str:
    '''Returns a .js file that will display the error message.
    '''

    jsText = errorMsg

    # Escape quotation marks.
    jsText = jsText.replace('"', r'\"')

    # Escape newlines.
    jsText = jsText.replace('\n', r'\n')

    jsText = 'console.error("' + jsText + '");'

    return jsText

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
