#-------------------------------------------------------------------
#  AppSetup.py
#
#  The AppSetup class.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the AppSetup class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.db.DatabaseMgr import DatabaseMgr
from ai.axe.web.config import DbConfig
from ai.axe.web.core import Log
from ..api.AuthMgr import AuthMgr
from ai.axe.util import StringUtil
from ai.axe.web.config import Config
from typing import Callable
from typing import Optional
from typing import Type
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class AppSetup:
  '''The setup for an Axe-based webapp.

  To use the Axe webapp functionality, you must create a subclass
  of AppSetup and register it using AppSetup.register().

  This must be done before using any other Axe web functionality.
  '''

  #-----------------------------------------------------------------
  # The AppSetup currently being used.
  setup: Optional['AppSetup'] = None

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new AppSetup.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def register(appSetup: 'AppSetup') -> None:
    '''Registers an appSetup for the current Axe app.

    @param appSetup An object that is a AppSetup subclass.
    '''

    AppSetup.setup = appSetup

    Log.setLogFileBaseName(appSetup.appName())

  #-----------------------------------------------------------------
  def appName(self) -> str:
    '''Returns the name of the app.

    This name is a camel-case string with no spaces.

    This name will be converted from camelCase to ALL_CAPS_AND_UNDERBARS
    to make environment variable names (such as "CROP_DOCTOR_HOME").

    @return A string that is the name of the app, such as "ram" or "cropDoctor".
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def rootPackageName(self) -> str:
    '''Returns the name of the root Python package.

    This name should be a string like "a.b.c.mypackage".

    The default package name is simply the app name.  If you have
    a different pacakge name, you should override this method to
    return that name.

    @return A string that is the name of the root package such as 'ai.axe' or
            'ram'.
    '''

    return self.appName()

  #-----------------------------------------------------------------
  def hasFrontEnd(self) -> bool:
    '''Returns whether this app has a web browser-based front end.

    If True, the files in src/webapp will be added to the distribution.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def webAuthMgrClass(self) -> Type[AuthMgr]:
    '''Returns the AuthMgr class that should be used by the webapp.

    @return A constructor function that can be called to create an object
            that is an AuthMgr subclass.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def databaseMgr(self) -> DatabaseMgr:
    '''Returns the DatabaseMgr object that should be used by the webapp.

    This method should always return the same object when called.

    This method will only be called after initWeb() or initCommandLine()
    is called.

    @return An object that is a DatabaseMgr subclass.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def sqlBase(self) -> type:
    '''Returns the base class used to make database classes.

    This method should always return the same object when called.

    This method may be called before initWeb() or initCommandLine()
    is called.

    @return An object that is a Sql Alchemy declarative_base().
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def initWeb(self, noisyErrors: bool) -> None:
    '''Carry out initialization of the webapp.

    In this method, you should at least:

      - Set up the Config file, adding any additional sections.
      - Read the LogConfig and call Log.initWebLog(logConfig, noisyErrors).
      - Read the DbConfig and create a DatabaseMgr.
      - Import any action classes, so they get registered.

    You may also do other initialization you require.

    @param noisyErrors If true, exceptions will be written to stderr.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def initCommandLine(self,
                      modifyDbConfigFunc: Callable[[DbConfig],
                                                   DbConfig]) -> None:
    '''Carry out initialization of a command line action.

    In this method, you should at least:

      - Set up the Config file, adding any additional sections.
      - Read the LogConfig and call Log.initCommandLog(logConfig).
      - Read the DbConfig and create a DatabaseMgr.
      - Import any AxeCommand classes, so they get registered.

    You may also do other initialization you require.

    @param modifyDbConfigFunc A function that the DbConfig should be passed
                              through.  The return value of this will be used
                              as the DbConfig.
    '''

    raise NotImplementedError()

  #----------------------------------------------------------------
  def initUnitTest(self) -> None:
    '''Carry out initialization of a unit test.

    In this method, you should at least:

      - Set up the Config file, adding any additional sections.
      - Read the DbConfig and create a DatabaseMgr.

    You may also do other initialization you require.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  @staticmethod
  def get() -> 'AppSetup':
    '''Returns the appSetup for the current Axe app.

    @return An object that is a AppSetup subclass.
    '''

    if AppSetup.setup:
      return AppSetup.setup
    else:
      msg = "Error.  The AppSetup.register() method has not yet been called "
      msg += "to register the AppSetup class for your Axe webapp."
      raise ValueError(msg)

  #-----------------------------------------------------------------
  @staticmethod
  def exists() -> bool:
    '''Returns True if an appSetup has been registered for the current Axe app.
    '''

    return bool(AppSetup.setup)

  #-----------------------------------------------------------------
  def appNameAllCaps(self) -> str:
    '''Returns the name of the app, in all caps and underbars.

    Do not override this method unless you know what you are doing.

    @return The string name of the app.
    '''

    lowerUnder = StringUtil.camelCaseToUnderscores(self.appName())
    upperUnder = lowerUnder.upper()

    return upperUnder

  #-----------------------------------------------------------------
  def initWebFull(self, noisyErrors: bool) -> None:
    '''Does a 'full' web init, including calling this app's initWeb().

    Do not override this method unless you know what you are doing.

    @param noisyErrors If true, exceptions will be written to stderr.
    '''

    Config.setEnvVarPrefix(self.appNameAllCaps())

    self.initWeb(noisyErrors)

  #-----------------------------------------------------------------
  def initCommandLineFull(self, modifyDbConfigFunc: Callable[[DbConfig],
                                                             DbConfig]) -> None:
    '''Does a 'full' command-line init, including calling initCommandLine().

    Do not override this method unless you know what you are doing.

    @param modifyDbConfigFunc A function that the DbConfig should be passed
                              through.  The return value of this will be used
                              as the DbConfig.
    '''

    Config.setEnvVarPrefix(self.appNameAllCaps())

    self.initCommandLine(modifyDbConfigFunc)

  #-----------------------------------------------------------------
  def initUnitTestFull(self) -> None:
    '''Does a 'full' unittest init, including calling this app's initUnitTest().

    Do not override this method unless you know what you are doing.
    '''

    # Have to import here in this function to avoid a circular include.
    # pylint: disable=import-outside-toplevel
    from ai.axe.build.unittest.AxeTestCase import AxeTestCase

    Config.setEnvVarPrefix(self.appNameAllCaps())

    self.initUnitTest()

    AxeTestCase.setDatabaseMgr(self.databaseMgr())
