#-------------------------------------------------------------------
#  TestWebDescTypescriptWriter.py
#
#  The TestWebDescTypescriptWriter module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the WebDescTypescriptWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from ai.axe.web.api.desc import WebDesc
from ai.axe.web.api.desc import WebDescTypescriptWriter
from ai.axe.web.api.desc import WebVerbDesc
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import Dict
from typing import List
from typing import Optional
from .WebDescTypescriptWriterStrings import WebDescTypescriptWriterStrings
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestWebDescTypescriptWriter(AxeSimpleTestCase):
  '''Unit test for the WebDescTypescriptWriter class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.

    # pylint: disable=unused-import

    # Import the test web action.
    #
    # Import here within the test function so the test action
    # doesn't get created during normal application use.
    from .diamond import SellDiamond # pylint: disable=import-outside-toplevel

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testTypescript(self) -> None:
    '''Test generating a Typescript doc for web verbs.
    '''

    # Shortening function names to save space.
    format = WebDescTypescriptWriter.formatVerbDict

    testStrings = WebDescTypescriptWriterStrings()
    allVerbs = WebDesc.allVerbs(False)

    # appraiseDiamondText
    expected = testStrings.appraiseDiamondTextDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'appraise', 'text')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "appraiseDiamondText")

    # appraiseDiamondMultiple
    expected = testStrings.appraiseDiamondMultipleDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'appraise', flavor='multiple')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "appraiseDiamondMultiple")

    # appraiseDiamondImage
    expected = testStrings.appraiseDiamondImageDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'appraise', flavor='image')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "appraiseDiamondImage")

    # appraiseDiamondPdf
    expected = testStrings.appraiseDiamondPdfDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'appraise', flavor='pdf')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "appraiseDiamondPdf")

    # appraiseDiamondRaw
    expected = testStrings.appraiseDiamondRawDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'appraise',
                            flavor='rawResponse')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "appraiseDiamondRaw")

    # sellDiamond
    expected = testStrings.sellDiamondDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'sell')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "sellDiamond")

    # listDiamond
    expected = testStrings.listDiamondDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'list')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "listDiamond")

    # buyDiamond
    expected = testStrings.buyDiamondDescText
    verbDict = self.subDict(allVerbs, 'diamond', 'buy', flavor='')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "buyDiamond")

    # uploadDiamond
    expected = testStrings.uploadDiamond
    verbDict = self.subDict(allVerbs, 'diamond', 'upload')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "uploadDiamond")

    # findBuyerDiamond
    expected = testStrings.findBuyerDiamond
    verbDict = self.subDict(allVerbs, 'diamond', 'findBuyer')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "findBuyerDiamond")

    # findPersonDiamond
    expected = testStrings.findPersonDiamond
    verbDict = self.subDict(allVerbs, 'diamond', 'findPerson')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "findPersonDiamond")

    # findPersonExistsDiamond
    expected = testStrings.findPersonExistsDiamond
    verbDict = self.subDict(allVerbs, 'diamond', 'findPerson', flavor='exists')
    actual = format(verbDict)
    self.assertEqual(expected, actual, "findPersonExistsDiamond")

    # All
    expected = testStrings.allDescText
    verbDict = self.subDict(allVerbs, 'diamond', None, None)
    actual = WebDescTypescriptWriter.formatVerbDict(verbDict)
    self.assertEqual(expected, actual, "all")

  #-----------------------------------------------------------------
  def testCreation(self) -> None:
    '''Tests creating an object.
    '''

    # Doesn't test the returned values.
    # Just tests that no exception gets thrown, and gets test coverage up.

    str(WebDescTypescriptWriter())

  #-----------------------------------------------------------------
  def subDict(self,
              verbsDict: Dict[str, List[WebVerbDesc]],
              noun: str,
              verb: Optional[str],
              flavor: Optional[str] = '') -> Dict[str, List[WebVerbDesc]]:
    '''Given a noun, verb, and flavor, finds the corresponding
    element in verbsDict and returns a dictionary with that
    element as the only entry.

    @param verbsDict A {noun : [verbs]} dictionary to copy from.
    @param noun A noun string to filter on.
    @param verb A verb string to filter on.
                If None, all verbs will be returned.
    @param flavor A flavor string to filter on.
                  If None, all flavors will be returned.
    '''

    subDict: Dict[str, List[WebVerbDesc]] = OrderedDict()

    for nounKey, webVerbList in verbsDict.items():

      if nounKey == noun:
        for webVerb in webVerbList:

          # Make sure verb and flavor are correct.
          if ((not verb or webVerb.verb == verb) and
              (flavor is None or webVerb.flavor == flavor)):
            subDict.setdefault(noun, []).append(webVerb)

    assert subDict, 'Unable to find verb in verbsDict'

    return subDict
