#-------------------------------------------------------------------
#  WebVerbDesc.py
#
#  The WebVerbDesc class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebVerbDesc class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .DocComment import DocComment
from .JsonResponseDesc import JsonResponseDesc
from .WebParam import WebParam
from ai.axe.classJson.jsonTypes import JsonObj
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.util import StringUtil
from typing import Any
from typing import Callable
from typing import List
from typing import Tuple
from typing import TypeVar
from typing import Union
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
AnyFunction = Callable[..., Any]

#-------------------------------------------------------------------
AnyFunctionType = TypeVar("AnyFunctionType", bound=AnyFunction)

#===================================================================
class WebVerbDesc:
  '''Description of a web URL verb.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               noun: str,
               verb: str,
               flavor: str,
               url: str,
               contextParams: List[Tuple[str, str]],
               requestParams: List[WebParam],
               responseClass: Union[type, JsonResponseDesc, None],
               publicApi: bool,
               internal: bool,
               authless: bool,
               dynamicParams: bool,
               docComment: DocComment) -> None:
    '''Creates a new WebVerbDesc.

    @param requestParams A list of WebParam that describe the parameters
                         to be passed into the action function.
    @param responseClass The class that will be returned from the function,
                         or a JsonResponseDesc object,
                         or 'text' to return plain text.
    @param flavor If a single verb has multiple flavors, this string
                  is the flavor to be associated with the decorated action
                  function.
    @param publicApi True if this should be included as a public API
                     call under the /api URL tree.
    @param internal True if action is meant to be called internally by the
                    browser-based front end.  False if the browser will not
                    call this.  If False, no Typescript stubs will be
                    generated for this action.
    @param authless True if the action should be allowed to be called
                    without logging in.  This is very dangerous,
                    and should be used sparingly.
    @param dynamicParams True if the params will be different with every
                         HTTP call, and should just be passed into a variable
                         called 'params' of type List[Tuple[str, str]].
                         Warning: this disables all error checking and
                         documentation generation for parameters.
    '''

    self.noun: str = noun
    self.verb: str = verb
    self.flavor: str = flavor
    self.url: str = url
    self.contextParams: List[Tuple[str, str]] = contextParams
    self.requestParams: List[WebParam] = requestParams
    self.responseClass: Union[type, JsonResponseDesc, None] = responseClass
    self.publicApi: bool = publicApi
    self.internal: bool = internal
    self.authless: bool = authless
    self.dynamicParams: bool = dynamicParams
    self.docComment: DocComment = docComment

  #-----------------------------------------------------------------
  def allClasses(self) -> List[JsonObj]:
    '''Returns all @ClassJsonClass-decorated classes used in this verb.
    '''

    jsonObjs: List[JsonObj] = []

    # Add the parameter classes.

    for webParam in self.requestParams:
      if (webParam.objClass is not None and
          JsonObjRegistry.classIsRegistered(webParam.objClass)):
        jsonObj = JsonObjRegistry.getForClass(webParam.objClass)
        jsonObjs.append(jsonObj)
        jsonObjs.extend(JsonObjRegistry.findAllClasses(jsonObj))

    # Add the response class.

    if isinstance(self.responseClass, JsonResponseDesc):
      jsonObjs.extend(self.responseClass.findAllClasses())

    elif (isinstance(self.responseClass, type) and
          JsonObjRegistry.classIsRegistered(self.responseClass)):
      jsonObj = JsonObjRegistry.getForClass(self.responseClass)
      jsonObjs.append(jsonObj)
      jsonObjs.extend(JsonObjRegistry.findAllClasses(jsonObj))

    return jsonObjs

  #-----------------------------------------------------------------
  def typescriptFunctionName(self) -> str:
    '''Returns the name of the typescript function for this web action.
    '''

    return StringUtil.initialLower(self.typescriptClassName())

  #-----------------------------------------------------------------
  def typescriptClassName(self) -> str:
    '''Returns the name of the typescript class for this web action.
    '''

    capital = StringUtil.capitalize

    return capital(self.verb) + capital(self.noun) + capital(self.flavor)

# TODO delete
#  #-----------------------------------------------------------------
#  def isContextParam(self, name: str) -> bool:
#    '''Returns true if there's a context parameter with the specified name.
#    '''
#
#    for paramName, unusedParamType in self.contextParams:
 #     if paramName == name:
 #       return True
#
#    return False

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['noun',
             'verb',
             'flavor',
             'url',
             'internal',
             'requestParams',
             'contextParams',
             'responseClass']

    return StringUtil.formatRepr(self, attrs)
