#-------------------------------------------------------------------
#  WebVerb.py
#
#  The WebVerb class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebVerb class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .DocComment import DocComment
from .DocParser import DocParser
from .JsonResponseDesc import JsonResponseDesc
from .WebDesc import WebDesc
from .WebParam import WebParam
from .WebVerbDesc import WebVerbDesc
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.web.core import AxeException
from ai.axe.util import StringUtil
from ai.axe.web.api import BinaryResponse
from ai.axe.web.api import DownloadResponse
from typing import Any
from typing import Callable
from typing import List
from typing import Tuple
from typing import TypeVar
from typing import Union
from werkzeug.wrappers import Response
import inspect
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
AnyFunction = Callable[..., Any]

#-------------------------------------------------------------------
AnyFunctionType = TypeVar("AnyFunctionType", bound=AnyFunction)

#===================================================================
class WebVerb:
  '''Description of a web URL verb.

  This is a decorator used like so:

  @WebVerb([WebParam(name='foo', type='bool'),
            WebParam(name='bar', type='int')],
            MyJsonClass)
  def myFunction(foo, bar):
    ...
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               paramList: List[WebParam],
               responseClass: Union[type, JsonResponseDesc, None],
               flavor: str = '',
               publicApi: bool = False,
               internal: bool = True,
               authless: bool = False,
               dynamicParams: bool = False) -> None:
    '''Creates a new WebVerb.

    @param paramList A list of WebParam that describe the parameters
                     to be passed into the action function.
    @param responseClass The class of the object that will be returned from
                         the function.  One of:
                         a @ClassJsonClass-decorated class,
                         a JsonResponseDesc object,
                         a str to return plain text,
                         a BinaryResponse,
                         a DownloadResponse, or
                         a werkzeug Response.

    @param flavor If a single verb has multiple flavors, this string
                  is the flavor to be associated with the decorated action
                  function.
    @param publicApi True if this should be included as a public API
                     call under the /api URL tree.
    @param internal True if action is meant to be called internally by the
                    browser-based front end.  False if the browser will not
                    call this.  If False, no Typescript stubs will be
                    generated for this action.
    @param authless True if the action should be allowed to be called
                    without logging in.  This is very dangerous,
                    and should be used sparingly.
    @param dynamicParams True if the params will be different with every
                         HTTP call, and should just be passed into a variable
                         called 'params' of type List[Tuple[str, str]].
                         Warning: this disables all error checking and
                         documentation generation for parameters.
    '''

    self.requestParams: List[WebParam] = paramList
    self.responseClass: Union[type, JsonResponseDesc, None] = responseClass
    self.flavor: str = flavor
    self.publicApi: bool = publicApi
    self.internal: bool = internal
    self.authless: bool = authless
    self.dynamicParams: bool = dynamicParams

  #-----------------------------------------------------------------
  def register(self, actionFunction: AnyFunctionType) -> AnyFunctionType:
    '''Registers the function as a web verb.

    @param actionFunction The function to be registered.
    '''

    self.validateResponseClass(actionFunction)

    moduleName = actionFunction.__module__
    moduleNameElems = moduleName.split('.')

    verb: str = moduleNameElems.pop(-1)
    noun: str = moduleNameElems.pop(-1)

    verb = StringUtil.initialLower(verb)

    # If the verb module name was verbNoun, remove the noun.
    if verb.lower().endswith(noun.lower()):
      verb = verb[:-len(noun)]

    url: str = noun + '/' + verb

    if self.flavor:
      url += '/' + self.flavor

    contextParams: List[Tuple[str,
                              str]] = self.findContextParams(actionFunction)

    errorMsg: str = self.checkParams(actionFunction, contextParams)
    if errorMsg:
      errorMsg = ('Error processing @WebVerb decoration on function ' +
                  self.functionName(actionFunction) + '\n' + errorMsg)
      raise AxeException(errorMsg)

    docString = ''
    if isinstance(actionFunction.__doc__, str):
      docString = actionFunction.__doc__

    docParser = DocParser()
    docComment: DocComment = docParser.parse(docString)

    desc = WebVerbDesc(noun,
                       verb,
                       self.flavor,
                       url,
                       contextParams,
                       self.requestParams,
                       self.responseClass,
                       self.publicApi,
                       self.internal,
                       self.authless,
                       self.dynamicParams,
                       docComment)

    responseClass = self.responseClass
    if isinstance(responseClass, JsonResponseDesc):
      responseClass.setJsonClassName(desc.typescriptClassName() + "Response")

    WebDesc.register(actionFunction, desc)

    return actionFunction

  #-----------------------------------------------------------------
  def __call__(self, actionFunction: AnyFunctionType) -> AnyFunctionType:
    '''Registers the function as a web verb.

    @param actionFunction The function to be registered.
    '''

    return self.register(actionFunction)

  #-----------------------------------------------------------------
  def validateResponseClass(self, actionFunction: AnyFunction):
    '''Throws an exception if the responseClass is invalid.
    '''

    responseClass = self.responseClass

    validResponseClasses = (None,
                            str,
                            BinaryResponse,
                            DownloadResponse,
                            Response)

    if not (responseClass in validResponseClasses or
            isinstance(responseClass, JsonResponseDesc) or
            (isinstance(responseClass, type) and
             JsonObjRegistry.classIsRegistered(responseClass))):

      functionName = self.functionName(actionFunction)

      valids = 'None, str, BinaryResponse, DownloadResponse, Response'
      msg = ('Error decorating @WebVerb function %s.  The responseClass ' +
             'must be one of: ' + valids + ', a JsonResponseDesc, ' +
             'or a class with a @ClassJsonClass decoration. Did you forget ' +
             'to add a @ClassJsonClass decoration to your response class? ' +
             'You passed in a %s type with value: %s')
      msg = msg % (functionName, type(responseClass), responseClass)

      raise AxeException(msg)

  #-----------------------------------------------------------------
  def isRequestParam(self, name: str) -> bool:
    '''Returns true if there's a request parameter with the specified name.
    '''

    for desc in self.requestParams:
      if desc.name == name:
        return True

    return False

  #-----------------------------------------------------------------
  def findContextParams(self,
                        actionFunction: AnyFunction) -> List[Tuple[str, str]]:
    '''Checks the specified function's parameters for context parameters.

    @param actionFunction The function to be checked.

    @return A list of (name, typeString) tuples.
    '''

    ret = []

    fullArgSpec = inspect.getfullargspec(actionFunction)
    args = fullArgSpec.args

    for arg in args:
      if self.isRequestParam(arg):
        continue

      elif arg == 'session':
        ret.append(('session', 'Session'))

      elif arg == 'authMgr':
        ret.append(('authMgr', 'AuthMgr'))

      elif arg == 'userInfo':
        ret.append(('userInfo', 'UserInfo'))

      elif arg == 'config':
        ret.append(('config', 'Config'))

      elif arg.endswith('Config'):
        ret.append((arg, StringUtil.capitalize(arg)))

      elif arg == 'request':
        ret.append(('request', 'Request'))

      elif arg == 'webWorkerRunner':
        ret.append(('webWorkerRunner', 'WebWorkerRunner'))

    return ret

  #-----------------------------------------------------------------
  def checkParams(self,
                  actionFunction: AnyFunction,
                  contextParams: List[Tuple[str, str]]) -> str:
    '''Checks that the specified function's parameter match this verb's.

    @param actionFunction The function to be checked.

    @return A string error message if there's a problem, or None if
            everything is OK.
    '''

    fullArgSpec = inspect.getfullargspec(actionFunction)
    args = fullArgSpec.args

    extra = []

    missing = []

    # Put all the names in the 'missing' list.
    for desc in self.requestParams:
      missing.append(desc.name)
    for paramName, unusedParamType in contextParams:
      missing.append(paramName)

    for arg in args:
      if arg in missing:
        missing.remove(arg)
      else:
        extra.append(arg)

    if self.dynamicParams:
      if missing or extra != ['params']:
        msg = 'A @WebVerb with dynamicParameters=True should have a single '
        msg += 'argument called "params" with type List[Tuple[str, str]].'
        return msg
      else:
        extra.remove('params')

    msg = ''
    if missing:
      msg += 'These @WebParam decorations do not match any function argument: '
      msg += ', '.join(missing)
    if missing and extra:
      msg += '\n'
    if extra:
      msg += 'These function arguments are missing a @WebParam decoration: '
      msg += ', '.join(extra)

    return msg

  #----------------------------------------------------------------
  def functionName(self, theFunction: AnyFunction) -> str:
    '''Returns the fully-qualified name of the specified function object.
    '''

    return theFunction.__module__ + '.' + theFunction.__name__ + '()'

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['noun', 'verb', 'flavor', 'requestParams', 'responseClass']

    return StringUtil.formatRepr(self, attrs)
