#-------------------------------------------------------------------
#  WebParam.py
#
#  The WebParam class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebParam class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.web.api.RequestParamEncoding import RequestParamEncoding
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
from typing import Union
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
# Magic value to indicate that there is no default value.
#
# We can't use 'None' because we want 'None' to be a valid default value.
#
noDefaultValue = 'NO_DEFAULT_VALUE_MAGIC_VALUE'

#===================================================================
class WebParam:
  '''Description of a parameter to a web action function.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               name: str,
               paramType: Union[type, List[type]],
               urlName: Optional[str] = None,
               optional: bool = False,
               defaultValue: object = noDefaultValue,
               objClass: Optional[type] = None) -> None:
    '''Creates a new WebParam.

    @param name The name of the function parameter this describes.
    @param paramType The type to be passed into the action function.
    @param urlName The name of the parameter in the URL, if different
                   from the 'name' parameter.
    @param optional If true, this parameter will be optional.  If not
                    provided by the HTTP requestor, it will be None.
    @param defaultValue If provided, this value will be used when
                        the HTTP requestor does not provide a value.
    @param objClass If the paramType is object or object list,
                    this must be supplied
                    to specify what class object is expected.
    '''

    theType, theClass = RequestParamEncoding.forType(paramType, objClass, name)

    self.name: str = name
    self.paramType: RequestParamEncoding = theType
    self.urlName: Optional[str] = urlName
    self.optional: bool = optional
    self.defaultValue: object = defaultValue
    self.objClass: Optional[type] = theClass

  #-----------------------------------------------------------------
  def effectiveUrlName(self) -> str:
    '''Returns the current effective value for urlName.
    '''

    if self.urlName:
      return self.urlName
    else:
      return self.name

  #-----------------------------------------------------------------
  def hasDefaultValue(self) -> bool:
    '''Returns true if this has a default value.
    '''

    return self.defaultValue != noDefaultValue

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['name', 'paramType']

    return StringUtil.formatRepr(self, attrs)
