#-------------------------------------------------------------------
#  WebDescTypescriptWriter.py
#
#  The WebDescTypescriptWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebDescTypescriptWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .JsonResponseDesc import JsonResponseDesc
from .WebVerbDesc import WebVerbDesc
from .WebParam import WebParam
from ai.axe.classJson import ClassJsonFieldDesc
from ai.axe.classJson import ClassJsonTypescriptWriter
from ai.axe.classJson.jsonTypes import JsonObj
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.util import StringUtil
from ai.axe.web.api.BinaryResponse import BinaryResponse
from ai.axe.web.api.DownloadResponse import DownloadResponse
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
from werkzeug.wrappers import Response
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WebDescTypescriptWriter:
  '''Writes a typescript page for a set of web verbs.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WebDescTypescriptWriter.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def formatVerbDict(verbDict: Dict[str, List[WebVerbDesc]]) -> str:
    '''Formats a [noun : list of verbs] dictionary.

    @param verbDict dictionary of {noun: list of verbs}
    '''

    # Initial value for text
    functionText = ''
    responseClassText = ''
    classJsonClassNames: List[str] = []
    importedClasses: List[str] = []

    for ignoredNoun, verbList in verbDict.items():

      items = WebDescTypescriptWriter.formatVerbs(verbList)
      if items[0]:
        functionText += items[0] + "\n\n"
      if items[1]:
        responseClassText += items[1] + "\n\n"
      classJsonClassNames.extend(items[2])
      importedClasses.extend(items[3])

    cjWriter = ClassJsonTypescriptWriter()

    # Build up the imports for ClassJson classes used in the actions.

    # Remove duplicates.
    importedClasses = list(set(importedClasses))

    importedClasses.sort()

    importStr = "import { %s } from './classJsonInterface';"

    importLines = [importStr % clazz for clazz in importedClasses]
    importedClassesText = '\n'.join(importLines)

    # Build up the register descs function for all the response classes.

    jsonObjs: List[JsonObj] = []
    for verbList in verbDict.values():
      for verb in verbList:

        if not verb.internal:
          continue

        clazz = verb.responseClass
        if isinstance(clazz, JsonResponseDesc):
          clazz = clazz.getJsonClass()

        if isinstance(clazz, type) and JsonObjRegistry.classIsRegistered(clazz):
          jsonObjs.append(JsonObjRegistry.getForClass(clazz))

    descs = cjWriter.writeRegisterDescs(jsonObjs, 0)

    # The register function for all the classes.
    register = cjWriter.writeRegister(classJsonClassNames, "Basic%s", 0)

    # Assemble the string.

    text = ''
    text += '\n'
    text += importedClassesText
    text += '\n'
    text += '\n'
    text += 'export type ErrorHandler = (response?: any) => void;\n'
    text += '\n'
    text += functionText
    text += '\n'
    text += '// The types below are private implementation classes.\n'
    text += '// Do not use them.\n'
    text += '// You should instead use the interfaces listed above.\n'
    text += '\n'
    text += responseClassText
    text += '\n'
    text += register
    text += '\n'
    text += descs

    return text

 #-----------------------------------------------------------------
  @staticmethod
  def formatVerbs(webVerbs: List[WebVerbDesc]) -> Tuple[str,
                                                        str,
                                                        List[str],
                                                        List[str]]:
    '''Formats a list of WebVerbDesc objects into strings.

    @param webVerbs A list of WebVerbDesc objects to be formatted.

    @return A (text, responseClassText, classJsonClassName, importedClasses)
            tuple of strings.
            The 'text' is the typescript code for the function.
            The 'responseClassText' is typescript to define a new class if
            the response is a JsonResponseDesc, or '' if the return class is
            a normal @ClassJsonClass.
            The 'classJsonClassNames' is the name to use for the generated
            JsonResponseClass class, as well as any normally created static
            @ClassJsonClass.
            The 'importedClasses' is all the ClassJson classes that need
            to be imported.
    '''

    text: str = ''
    responseClassText: str = ''
    classJsonClassNames: List[str] = []
    importedClasses: List[str] = []

    # Loop through all webVerbs and concatonate the output strings
    for webVerb in webVerbs:

      if not webVerb.internal:
        continue

      items = WebDescTypescriptWriter.formatVerb(webVerb)
      text += items[0]
      responseClassText += items[1]
      if items[2] is not None and items[2]:
        classJsonClassNames.append(items[2])

      importedClasses.extend([x.desc.className for x in webVerb.allClasses()])

    return text, responseClassText, classJsonClassNames, importedClasses

  #-----------------------------------------------------------------
  @staticmethod
  def formatVerb(webVerb: WebVerbDesc) -> Tuple[str,
                                                str,
                                                Optional[str],
                                                str]:
    '''Formats a WebVerbDesc object into a string.

    @param webVerb The verb to format.

    @return A (text,
               responseClassText,
               responseClassName,
               exampleText)
            tuple of strings.
            The 'text' is the typescript code for the function.
            The 'responseClassText' is typescript to define a new class if
            the response is a JsonResponseDesc, or '' if the return class is
            a normal @ClassJsonClass.
            The 'responseClassName' is the name to use for the generated
            JsonResponseClass class, or None if the return class is a normal
            @ClassJsonClass.
            The 'exampleText' is a plain text example of how to call this
            function in typescript.
    '''

    # Shortening of the capitalize function to make the string more concise.
    capitalize = StringUtil.capitalize

    # Set variables that will be used in the text string.
    # First we do the simple ones.
    noun = webVerb.noun
    verb = webVerb.verb
    flavor = webVerb.flavor
    params = webVerb.requestParams
    nounVerbFlavor = verb + capitalize(noun) + capitalize(flavor)

    # Before we go further, return the empty string if flavor='admin'
    if flavor == 'admin':
      return '', '', None, ''

    # More complex substrings are function calls.
    # Abbreviate this name for line length purposes
    thisClass = WebDescTypescriptWriter

    # Url string
    url = thisClass.getUrlString(noun, verb, flavor)

    # parameter string for the function
    functionArgs, paramAdds, fileAdds = thisClass.getParamStrings(params)

    # The Typescript class if the return is a JsonResponseDesc.
    interfaceText = ''
    responseClassText: str = ''
    responseClassName: Optional[str] = None

    if isinstance(webVerb.responseClass, JsonResponseDesc):

      jsonDesc = webVerb.responseClass
      cjWriter = ClassJsonTypescriptWriter()
      className = webVerb.typescriptClassName() + "Response"
      superclassNames: List[str] = []

      interfaceText = cjWriter.writeInterface(className,
                                              superclassNames,
                                              jsonDesc.fields,
                                              '',
                                              0)
      interfaceText += "\n"
      responseClassText = cjWriter.writeClass(className,
                                              '',
                                              'Basic%s',
                                              jsonDesc.fields,
                                              '',
                                              0)
      responseClassName = className

    # Success handler parameter (if applicable)
    (itemSuccessHandlerText,
     successParam,
     errorParam,
     successText) = thisClass.getSuccessText(webVerb)

    # The typescript function.

    # First, build up the declaration.

    # The actual generated typescript code for the function.
    functionText = 'export function ' + nounVerbFlavor + '('

    # An example of calling the function.
    exampleText = nounVerbFlavor + '('

    paramIndent = ' ' * len(functionText)
    exampleParamIndent = ' ' * len(exampleText)

    if functionArgs:
      functionText += (',\n' + paramIndent).join(functionArgs)
      exampleText += (',\n' + exampleParamIndent).join(functionArgs)

    if functionArgs and successParam:
      functionText += ',\n' + paramIndent
      exampleText += ',\n' + exampleParamIndent

    if successParam:
      functionText += successParam
      exampleText += 'successHandler?'

    if (functionArgs or successParam) and errorParam:
      functionText += ',\n' + paramIndent
      exampleText += ',\n' + exampleParamIndent

    if errorParam:
      functionText += errorParam
      exampleText += 'errorHandler?'

    functionText += ") : [Observable<any>, Subscription]\n"
    exampleText += ')'

    # Add the function implementation.

    # If we're uploading files, need to add extra arguments to the
    # createRequest() call.
    fileArgs = ', "POST", formData' if fileAdds else ''

    functionText += '''\
{
  let params: RequestParams = new RequestParams();'''

    if paramAdds:
      functionText += '\n\n  ' + '\n  '.join(paramAdds)

    if fileAdds:
      functionText += '\n\n  let formData = new FormData();'
      functionText += '\n\n  ' + '\n  '.join(fileAdds)

    functionText += '''

  let actionUrl = ''' + url + ''';

  let observable: Observable<any> = axeHttp.createRequest(actionUrl, params''' + fileArgs + ''');
  let subscription: Subscription = null;
''' + successText + '''

  return [observable, subscription];
}

'''

    text = itemSuccessHandlerText + interfaceText + functionText

    exampleText = thisClass.example(webVerb,
                                    exampleText,
                                    successParam,
                                    responseClassName)

    return text, responseClassText, responseClassName, exampleText

  #----------------------------------------------------------------
  @staticmethod
  def example(webVerb: WebVerbDesc,
              functionCallText: str,
              successParam: Optional[str],
              responseClassName: Optional[str]) -> str:
    '''Returns a plain text example of how to call the specified verb function.

    @param webVerb The function to generate an example for.
    @param functionCallText A string with an example of how to call the
                            actual function.
    @param successParam A string with the success handler parameter.
    @param responseClassName The name of the response class if it is a
                             JsonResponseDesc.
    '''

    capitalize = StringUtil.capitalize

    noun = webVerb.noun
    verb = webVerb.verb
    flavor = webVerb.flavor
    nounVerbFlavor = verb + capitalize(noun) + capitalize(flavor)

    # Indent 2 spaces.
    functionCallText = '  ' + functionCallText.replace('\n', '\n  ').strip()

    # Build up the response param information.

    exampleResponseParam = ''
    exampleResponseFields = []
    if successParam:
      responseType = ''
      responseTypeImport = ''
      properties: List[ClassJsonFieldDesc] = []
      responseClass = webVerb.responseClass

      if webVerb.responseClass is None:
        exampleResponseParam = ''

      else:

        if webVerb.responseClass == str:
          responseType = 'string'
        elif isinstance(responseClass, JsonResponseDesc):
          assert responseClassName is not None
          responseType = responseClassName
          responseTypeImport = 'webActions'
          properties = responseClass.fields
        elif (isinstance(responseClass, type) and
              JsonObjRegistry.classIsRegistered(responseClass)):
          jsonObj = JsonObjRegistry.getForClass(responseClass)
          responseType = '' + jsonObj.desc.className
          responseTypeImport = 'classJsonInterface'
          properties = jsonObj.desc.properties

        if properties:
          propertyNamespace = ''

          for propDesc in properties:
            line = (propDesc.name + ': ' +
                    propDesc.jsonType.toTypescriptLabel(propertyNamespace))
            exampleResponseFields.append(line)

        exampleResponseParam = 'response: ' + responseType

    exampleText = '''Call this web action like so:

  import { ''' + nounVerbFlavor  + ''' } from '../gen/webActions';
'''

    if successParam and responseTypeImport:
      importLine = "  import { %s } from '../gen/%s';"
      importLine = importLine %  (responseType, responseTypeImport)
      exampleText += importLine + '\n'

    exampleText += '\n' + functionCallText + '\n'

    if successParam:
      exampleText += '''
where successHandler is a function you write like:

  handleSuccess(''' + exampleResponseParam + ''') : void
  {
  }
'''

    if exampleResponseFields:
      exampleText += '''
and the 'response' object has the following fields:

  ''' + '\n  '.join(exampleResponseFields) + '''
'''

    return exampleText

  #-----------------------------------------------------------------
  @staticmethod
  def getUrlString(noun: str, verb: str, flavor: str) -> str:
    '''Helper function that gets the URL string.
    '''

    url = "/app/" + noun + "/" + verb
    if flavor:
      url += "/" + flavor

    url = "'" + url + "'"

    return url

  #-----------------------------------------------------------------
  @staticmethod
  def getParamStrings(params: List[WebParam]) -> Tuple[List[str],
                                                       List[str],
                                                       List[str]]:
    '''Returns the parameter passing strings.

    @param params A list of WebParam objects.

    @return A ([str], [str], [str], [str]) tuple:
            (1) Declaration of function arguments.
            (2) Code to add parameters to a RequestParam object.
            (3) Code to add File objects to a FormData object.
    '''

    # Declaration of function arguments.  List of strings.
    functionArgDecls = []

    # Code to add parameters to a RequestParam object.  List of strings.
    paramAdds = []

    # Code to add File objects to a FormData object.
    fileAdds = []

    for param in params:

      paramUrlName = param.effectiveUrlName()

      decl = (paramUrlName + ': ' +
              param.paramType.typescriptLabel('', param.objClass))
      functionArgDecls.append(decl)

      if not param.paramType.isFile():

        code = 'params.add("PARAM", PARAM);'
        code = code.replace("PARAM", paramUrlName)
        paramAdds.append(code)

      else:

        code = 'formData.append("PARAM", PARAM);'
        if param.paramType.isList():
          code = 'axeHttp.appendFileList(formData, "PARAM", PARAM);'

        code = code.replace("PARAM", paramUrlName)
        fileAdds.append(code)

    return functionArgDecls, paramAdds, fileAdds

  #-----------------------------------------------------------------
  @staticmethod
  def getSuccessText(webVerb: WebVerbDesc) -> Tuple[str, str, str, str]:
    '''Success/error handler interface, function parameter, and implementation.

    All empty strings are returned when there is no success handler.
    '''

    thisClass = WebDescTypescriptWriter

    if not thisClass.hasResponse(webVerb):

      # HttpClient return a "cold" observable, so we must subscribe,
      # or the HTTP request will never fire.
      successText = '''
  subscription = observable.subscribe();
'''

      return '', '', '', successText


    classNameRoot = webVerb.typescriptClassName()
    successHandlerName = classNameRoot + 'SuccessHandler'

    responseParam = thisClass.successResponseParamDecl(webVerb)

    handlerInterface = '''\
export type ''' + successHandlerName + ''' =
  (''' + responseParam + ''') => void;

'''
    successParam = 'successHandler?: ' + successHandlerName
    errorParam = 'errorHandler?: ErrorHandler'

    # This is the implementation code that calls the success handler.
    successText = '''
  if (successHandler)
  {
    if (!errorHandler)
    {
      errorHandler = axeHttp.defaultOnError;
    }

    subscription = observable.subscribe(successHandler,
                                        errorHandler);
  }'''

    return handlerInterface, successParam, errorParam, successText

  #-----------------------------------------------------------------
  @staticmethod
  def successResponseParamDecl(webVerb: WebVerbDesc) -> str:
    '''Declaration of the 'response' parameter that the success handler takes.

    Will be an empty string if the web verb returns None, so there is no
    'response' parameter passed into the success handler.
    '''

    thisClass = WebDescTypescriptWriter

    if webVerb.responseClass is None:
      return ''

    else:

      responseType = thisClass.successResponseParamType(webVerb)
      return "response: " + responseType

  #-----------------------------------------------------------------
  @staticmethod
  def successResponseParamType(webVerb: WebVerbDesc) -> str:
    '''The type of the 'response' parameter that the success handler takes.
    '''

    if webVerb.responseClass == str:
      return 'string'

    elif isinstance(webVerb.responseClass, JsonResponseDesc):
      classNameRoot = webVerb.typescriptClassName()
      return classNameRoot + 'Response'

    elif (isinstance(webVerb.responseClass, type) and
          JsonObjRegistry.classIsRegistered(webVerb.responseClass)):
      jsonObj = JsonObjRegistry.getForClass(webVerb.responseClass)
      className = jsonObj.desc.className
      className = StringUtil.capitalize(className)
      return className

    else:
      raise AssertionError("Impossible verb:" + str(webVerb))

  #-----------------------------------------------------------------
  @staticmethod
  def hasResponse(webVerb: WebVerbDesc) -> bool:
    '''Returns true if the web verb has a response function.

    Most verbs have response functions, but some (like a download) do not.
    '''

    responseClass = webVerb.responseClass

    if responseClass in (DownloadResponse, BinaryResponse, Response):
      return False

    elif (responseClass is None or
          responseClass == str or
          isinstance(responseClass, JsonResponseDesc) or
          JsonObjRegistry.classIsRegistered(responseClass)):
      return True

    else:
      msg = 'Unknown webVerb.responseClass: %s' % webVerb.responseClass
      raise ValueError(msg)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
