#-------------------------------------------------------------------
#  WebDescAppDocWriter.py
#
#  The WebDescAppDocWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebDescAppDocWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .WebVerbDesc import WebVerbDesc
from ai.axe.web.api.BinaryResponse import BinaryResponse
from ai.axe.web.api.DownloadResponse import DownloadResponse
from typing import Dict
from typing import List
from werkzeug.wrappers import Response
from .JsonResponseDesc import JsonResponseDesc
from ai.axe.classJson import ClassJsonSchemaWriter
from ai.axe.util import StringUtil
from .WebDescTypescriptWriter import WebDescTypescriptWriter
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WebDescAppDocWriter:
  '''Writes a documentation page for a set of web verbs.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WebDescAppDocWriter.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def formatVerbDict(verbDict: Dict[str, List[WebVerbDesc]]) -> str:
    '''Formats a [noun : list of verbs] dictionary.
    '''

    text = 'Web Action Functions'
    text += '\n\n'

    for noun, verbList in verbDict.items():

      allNotInternal = all([(not x.internal) for x in verbList])
      if allNotInternal:
        continue

      text += noun
      for verbDesc in verbList:

        if not verbDesc.internal:
          continue

        docComment = verbDesc.docComment
        verbStr = verbDesc.verb
        if verbDesc.flavor:
          verbStr += '/' + verbDesc.flavor
        text += '\n\n  ' + verbStr + '\n\n  '
        for commentLine in docComment.shortDesc:
          text += commentLine

        text += '\n\n'

        text += '    Params:\n'
        for paramDesc, paramDoc in zip(verbDesc.requestParams,
                                       docComment.params.values()):

          text += ('      ' + paramDesc.effectiveUrlName() + '  ' +
                   paramDesc.paramType.label(paramDesc.objClass) + '  ' +
                                                        paramDoc + '\n')

        paramObjClasses: List[type] = []
        for paramDesc in verbDesc.requestParams:
          if paramDesc.objClass and paramDesc.objClass not in paramObjClasses:
            paramObjClasses.append(paramDesc.objClass)

        text += '    Param clases:\n'
        for objClass in paramObjClasses:
          text += ClassJsonSchemaWriter().writeClass(objClass, 3) + "\n"

        text += '    Response:\n'
        if verbDesc.responseClass is None:
          text += (' ' * 6) + 'Response has no data'
        elif verbDesc.responseClass == str:
          text += (' ' * 6) + 'Plain text'
        elif verbDesc.responseClass in (BinaryResponse, DownloadResponse):
          assert isinstance(verbDesc.responseClass, type)
          text += (' ' * 6) + verbDesc.responseClass.__name__
        elif verbDesc.responseClass == Response:
          text += (' ' * 6) + 'Raw HTTP response'
        elif isinstance(verbDesc.responseClass, JsonResponseDesc):
          fields = verbDesc.responseClass.fields
          text += ClassJsonSchemaWriter().writeFields(None, fields, 3)
        else:
          text += ClassJsonSchemaWriter().writeClass(verbDesc.responseClass, 3)

        text += '\n\n'
        text += '    Typescript\n\n'
        example = WebDescTypescriptWriter.formatVerb(verbDesc)[3]
        example = '      ' + example.replace('\n', '\n      ').strip()
        text += example

      text += '\n\n\n'

    return text

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
