#-------------------------------------------------------------------
#  WebDescApiDocWriter.py
#
#  The WebDescApiDocWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebDescApiDocWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .WebVerbDesc import WebVerbDesc
from ai.axe.web.api.BinaryResponse import BinaryResponse
from ai.axe.web.api.DownloadResponse import DownloadResponse
from typing import Dict
from typing import List
from werkzeug.wrappers import Response
from .JsonResponseDesc import JsonResponseDesc
from ai.axe.classJson import ClassJsonSchemaWriter
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WebDescApiDocWriter:
  '''Writes a documentation page for a set of web verbs.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WebDescApiDocWriter.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def formatVerbDict(verbDict: Dict[str, List[WebVerbDesc]]) -> str:
    '''Formats a [noun : list of verbs] dictionary.

    @param webDict The list of verbs to format.

    @return An HTML string to display to the user.
    '''

    cls = WebDescApiDocWriter


    tocHtml = cls.createToc(verbDict)

    verbHtmls: List[str] = []
    for noun, verbList in verbDict.items():

      for verb in verbList:

        verbHtmls.append(cls.createVerbSection(noun, verb))

    verbHtml = '\n\n'.join(verbHtmls)

    text = '''
<html>
<head>
  <title>Web API Documentation</title>
  <style>
    %s  
  </style>
</head>
<body>

<header>
</header>

<div class="mainbody">

  <h1>Web API Documentation</h1>

  <p>The Web API is made up of web actions.  Each action has a URL,
     takes HTTP parameters, and returns JSON data.  You are allowed
     to call these actions using either HTTP GET or POST.
  </p>

  <h3>Table of Contents</h3>

  <p>Each action URL consists of a noun, verb, and optional flavor.
     Here are the available actions, grouped by noun:
  </p>

  %s

  <h3>Parameters</h3>

  <p>Parameters can be one of the following types.
     This table shows the HTTP encoding to use for each type.
  </p>

  <table class="narrowTable">
    <thead>
      <tr>
        <th>Type</th>
        <th>Description</th>
        <th>Example HTTP parameter called 'foo'</th>
      </tr>
    </thead>
    <tbody>
      <tr>
        <td>boolean</td>
        <td>The string 'true' or 'false'.</td>
        <td>foo=true</td>
      </tr>
      <tr>
        <td>int</td>
        <td>An integer.</td>
        <td>foo=1001</td>
      </tr>
      <tr>
        <td>float</td>
        <td>A floating point number.</td>
        <td>foo=3.141592</td>
      </tr>
      <tr>
        <td>date</td>
        <td>A string of the format: YYYY-MM-DD</td>
        <td>foo=2019-03-15</td>
      </tr>
      <tr>
        <td>datetime</td>
        <td>A string of the format: YYYY-MM-DDThh:mm:ss.ssssZ
            Datetimes are in the UTC time zone.</td>
        <td>foo=2019-03-15T06:35:19.523Z</td>
      </tr>
      <tr>
        <td>duration</td>
        <td>An integer number of milliseconds</td>
        <td>foo=3600000</td>
      </tr>
      <tr>
        <td>string</td>
        <td>Simply the string itself.</td>
        <td>foo=blue</td>
      </tr>
      <tr>
        <td>list</td>
        <td>The parameter name repeated once for each item in the list.</td>
        <td>foo=blue&foo=red&foo=orange</td>
      </tr>
    </tbody>
  </table>

  %s

</div><!-- .mainbody -->

<footer></footer>
</body>
</html>
'''

    style = '''
body
{
  font-family: Helvetica, Arial, sans-serif;
  margin: 0;
}

/* Header/mainbody/footer formatting. ***************************** */

header
{
  background: #f5f6f7;
  height: 40px;
}

.mainbody
{
  margin-left: 8px;
  margin-right: 8px;
}

footer
{
  background: #f5f6f7;
  height: 40px;
}

/* Table formatting. ************************************************** */

table
{
  width: 100%;
  border: 1px solid #cececc;
  border-radius: 3px;
  border-spacing: 0;
}

thead>tr>th 
{
  color: #4b4f56;
  background: #f5f6f7;
  padding: 8px 12px;
  border-bottom: 1px solid #c9ced4;
  box-shadow: 0 1px 1px rgba(0, 0, 0, .05);
  border-left: 1px solid rgba(0, 0, 0, .1);
}

tr>td
{
  color: #4b4f56;
  border-left: 1px solid rgba(0, 0, 0, .1);
  padding: 8px;
}

tr>td:first-child
{
  border-left: 0;
}

tbody>tr:nth-child(even)>td
{
    background-color: #f5f6f7;
}

.narrowTable
{
  width: 70%;
  margin-left: 15%;
  margin-right: 15%
}

h1, h2, h3, h4, h5, h6
{
  color: #1d2129;
}


.verbLabel:after
{
  content:' ';
  display:block;
  border: 0.5px solid #777;
}
'''
    text = text % (style, tocHtml, verbHtml)

    return text

  #----------------------------------------------------------------
  @staticmethod
  def createToc(verbDict: Dict[str, List[WebVerbDesc]]) -> str:

    cls = WebDescApiDocWriter

    tocRows = []

    for noun, verbList in verbDict.items():

      description = ''
      nounRow = cls.createTocNounRow(noun, description)

      verbRows = []
      for verbDesc in verbList:

        docComment = verbDesc.docComment
        verbStr = verbDesc.verb
        if verbDesc.flavor:
          verbStr += '/' + verbDesc.flavor

        description = docComment.shortDesc

        verbRow = cls.createTocVerbRow(noun, verbStr, description)
        verbRows.append(verbRow)

      tocRows.append(nounRow)
      tocRows.extend(verbRows)

    return cls.createTocTable(tocRows)

  #----------------------------------------------------------------
  @staticmethod
  def createTocTable(tocRows: List[str]) -> str:
    '''Returns the HTML for the table of contents.
    '''

    text = '''
<table class="narrowTable">
  <thead>
    <tr><th>Noun</th><th>Verbs</th></tr>
  </thead>
  <tbody>
  %s
  </tbody>
</table>
'''

    nounText = '\n  ' + '\n  '.join(tocRows)
    text = text % nounText

    return text

  #----------------------------------------------------------------
  @staticmethod
  def createTocNounRow(noun: str, description: str) -> str:
    '''Returns the HTML for a single row of the table of contents.
    '''

    text = '<tr><td>%s</td><td>%s</td></tr>'
    text = text % (noun, description)

    return text

  #----------------------------------------------------------------
  @staticmethod
  def createTocVerbRow(noun: str, verb: str, description: str) -> str:
    '''Returns the HTML for a single row of the table of contents.
    '''

    anchor = noun + '_' + verb.replace('/', '_')

    text = '<tr><td></td><td><a href="#%s">%s</a>: %s</td></tr>'
    text = text % (anchor, verb, description)

    return text

  #----------------------------------------------------------------
  @staticmethod
  def createVerbSection(noun: str, verbDesc: WebVerbDesc) -> str:

    cls = WebDescApiDocWriter


    docComment = verbDesc.docComment
    verbStr = verbDesc.verb
    if verbDesc.flavor:
      verbStr += '/' + verbDesc.flavor

    anchor = noun + '_' + verbStr.replace('/', '_')

    descLines = [docComment.shortDesc] + docComment.longDesc
    description: str = '<p>' + '</p>\n<p>'.join(descLines) + '</p>'


    paramRows = []
    for param in verbDesc.requestParams:

      name = param.effectiveUrlName()
      paramType = param.paramType.label(param.objClass)
      paramDescStr = ""
      if param.name in docComment.params:
        paramDescStr = docComment.params[param.name]

      paramRow = cls.createParamRow(name, paramType, paramDescStr)
      paramRows.append(paramRow)

    paramsTable = cls.createParamsTable(paramRows)


    paramObjClasses: List[type] = []
    for paramDesc in verbDesc.requestParams:
      if paramDesc.objClass and paramDesc.objClass not in paramObjClasses:
        paramObjClasses.append(paramDesc.objClass)

    paramClasses = ''
    for objClass in paramObjClasses:
      text = ClassJsonSchemaWriter().writeClass(objClass, 3)
      text = cls.escapeHtml(text)
      text = '<pre>\n' + text + '\n</pre>\n'
      paramClasses += text
    if not paramClasses:
      paramClasses = '&lt;None>'

    response = ''
    if verbDesc.responseClass is None:
      response = (' ' * 6) + 'Response has no data'
    elif verbDesc.responseClass == str:
      response = (' ' * 6) + 'Plain text'
    elif verbDesc.responseClass in (BinaryResponse, DownloadResponse):
      assert isinstance(verbDesc.responseClass, type)
      response = (' ' * 6) + verbDesc.responseClass.__name__
    elif verbDesc.responseClass == Response:
      response = (' ' * 6) + 'Raw HTTP response'
    elif isinstance(verbDesc.responseClass, JsonResponseDesc):
      fields = verbDesc.responseClass.fields
      response = ClassJsonSchemaWriter().writeFields(None, fields, 3)
      response = cls.escapeHtml(response)
      response = '<pre>' + response + '</pre>'
    else:
      response = ClassJsonSchemaWriter().writeClass(verbDesc.responseClass, 3)
      response = cls.escapeHtml(response)
      response = '<pre>' + response + '</pre>'

    text = '''
<h2 class="verbLabel"><a id="%s">%s</a></h2>

%s

<p><span style="font-weight:bold">URL:</span> %s</p>

<h3>Parameters</h3>

%s

<h3>JSON objects used in parameters</h3>

%s

<h3>Response</h3>

%s
'''

    text = text % (anchor, verbDesc.url, description, verbDesc.url,
                   paramsTable, paramClasses, response)

    return text

  #----------------------------------------------------------------
  @staticmethod
  def createParamsTable(paramRows: List[str]) -> str:
    '''Returns the HTML for a params table.
    '''

    text = '''
<table>
  <thead>
    <tr><th>Parameter</th><th>Type</th><th>Description</th></tr>
  </thead>
  <tbody>
  %s
  </tbody>
</table>
'''

    paramsText = '\n  ' + '\n  '.join(paramRows)
    text = text % paramsText

    return text

  #----------------------------------------------------------------
  @staticmethod
  def createParamRow(name: str, paramType: str, description: str) -> str:
    '''Returns the HTML for a single row of a parameters table.
    '''

    text = '<tr><td>%s</td><td>%s</td><td>%s</td></tr>'
    text = text % (name, paramType, description)

    return text

  #----------------------------------------------------------------
  @staticmethod
  def escapeHtml(text: str) -> str:

    text = text.replace('<', '&lt;')

    return text

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
