#-------------------------------------------------------------------
#  WebDesc.py
#
#  The WebDesc class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebDesc class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .WebVerbDesc import WebVerbDesc
from ai.axe.classJson.jsonTypes import JsonObj
from ai.axe.web.core import AxeException
from collections import OrderedDict
from typing import Any
from typing import Callable
from typing import Dict
from typing import Iterable
from typing import List
from typing import Optional
import operator
#
# Import statements go above this line.
#-------------------------------------------------------------------

AnyFunction = Callable[..., Any]

#===================================================================
class WebDesc:
  '''Global access to Webapi functions.
  '''

  #-----------------------------------------------------------------
  # Mapping of action functions to descriptions.
  actionDescs: Dict[AnyFunction, WebVerbDesc] = OrderedDict()

  #-----------------------------------------------------------------
  # Mapping of URLs to action functions.
  appUrlFunctions: Dict[str, AnyFunction] = OrderedDict()

  #-----------------------------------------------------------------
  # Mapping of /api URLs to action functions.
  apiUrlFunctions: Dict[str, AnyFunction] = OrderedDict()

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WebDesc.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def register(actionFunction: AnyFunction, webVerb: WebVerbDesc) -> None:
    '''Registers an action function.

    @param actionFunction The function to be registered.
    @param webVerb The description of the function.
    '''

    WebDesc.actionDescs[actionFunction] = webVerb
    WebDesc.appUrlFunctions[webVerb.url] = actionFunction

    if webVerb.publicApi:
      WebDesc.apiUrlFunctions[webVerb.url] = actionFunction

  #-----------------------------------------------------------------
  @staticmethod
  def getForFunction(actionFunction: AnyFunction) -> WebVerbDesc:
    '''Returns the description for the specified function.
    '''

    if actionFunction not in WebDesc.actionDescs:
      msg = 'Error:  function not registered: %s' % actionFunction
      raise AxeException(msg)

    return WebDesc.actionDescs[actionFunction]

  #-----------------------------------------------------------------
  @staticmethod
  def allUrls(publicApi: bool) -> Iterable[str]:
    '''Returns all available URLs.

    @param publicApi Whether urls for all app actions are to be included,
                     or only the public API actions.
    '''

    if publicApi:
      return WebDesc.apiUrlFunctions.keys()
    else:
      return WebDesc.appUrlFunctions.keys()

  #-----------------------------------------------------------------
  @staticmethod
  def findAction(publicApi: bool, url: str) -> Optional[AnyFunction]:
    '''Returns the function whose URL matches the specified URL.

    @param publicApi Whether urls for all app actions are to be comsidered,
                     or only the public API actions.
    '''

    if publicApi:
      return WebDesc.apiUrlFunctions.get(url, None)
    else:
      return WebDesc.appUrlFunctions.get(url, None)

  #-----------------------------------------------------------------
  @staticmethod
  def allVerbs(publicApi: bool) -> Dict[str, List[WebVerbDesc]]:
    '''Return a [noun : list of verbs] dictionary.

    @param publicApi Whether urls for all app actions are to be included,
                     or only the public API actions.
    '''

    # Make a list of all nouns that have verbs.

    allNouns: List[str] = []
    for webVerb in WebDesc.actionDescs.values():

      if publicApi and not webVerb.publicApi:
        continue

      noun = webVerb.noun
      if noun not in allNouns:
        allNouns.append(noun)

    allNouns.sort()

    # Make a list of verbs that go with each noun.

    verbDict: Dict[str, List[WebVerbDesc]] = OrderedDict()
    for noun in allNouns:
      verbDict[noun] = []

    for webVerb in WebDesc.actionDescs.values():

      if publicApi and not webVerb.publicApi:
        continue

      noun = webVerb.noun
      verbDict[noun].append(webVerb)

    # Remove any nouns with no verbs.

    for noun in list(verbDict.keys()):
      if not verbDict[noun]:
        del verbDict[noun]
      else:
        verbDict[noun].sort(key=operator.attrgetter('url'))

    return verbDict

  #-----------------------------------------------------------------
  @staticmethod
  def allClasses() -> List[JsonObj]:
    '''Return a list of all @ClassJsonClass-decorated classes used in actions.
    '''

    classes: List[JsonObj] = []

    for webVerb in WebDesc.actionDescs.values():
      classes.extend(webVerb.allClasses())

    # Remove duplicates, preserving list order.

    oldClasses: List[JsonObj] = classes
    classes = []

    for clazz in oldClasses:
      if clazz not in classes:
        classes.append(clazz)

    return classes
